"""
Thread-safe and process-safe caching decorators.

Provides two main decorators:
1. @synchronized_cache - Thread-safe cache with deduplication (no thundering herd)
2. @shared_process_cache - Cross-process shared cache using multiprocessing.Manager

Both prevent duplicate expensive computations when called concurrently.
"""

import threading
import functools
from typing import Callable
from multiprocessing import Manager
from cachetools import LRUCache


def synchronized_cache(maxsize: int = 128):
    """
    Thread-safe LRU cache decorator that prevents duplicate function calls.

    Unlike functools.lru_cache, this decorator ensures that only ONE thread
    computes the result for a given set of arguments, even when called
    concurrently. Other threads wait for the first thread to complete.

    This solves the "thundering herd" problem where multiple threads all
    start computing the same expensive result before any of them finish.

    Args:
        maxsize: Maximum number of cached results to store

    Example:
        >>> @synchronized_cache(maxsize=10)
        ... def add(a, b):
        ...     return a + b
        >>> result = add(2, 3)
        >>> result
        5
        >>> add(2, 3)  # Cached result
        5
        >>> info = add.cache_info()
        >>> info['maxsize']
        10
        >>> info['size']
        1
        >>> add.cache_clear()
        >>> add.cache_info()['size']
        0

    Process-pool safe: Yes (each process gets independent cache)
    Thread-pool safe: Yes (prevents duplicate work across threads)
    """
    def decorator(func: Callable) -> Callable:
        cache = LRUCache(maxsize=maxsize)
        computing = {}  # Maps cache_key -> threading.Event
        cache_lock = threading.RLock()

        @functools.wraps(func)
        def wrapper(*args, **kwargs):
            # Create cache key
            key = args + tuple(sorted(kwargs.items()))

            # Fast path: check cache without lock
            if key in cache:
                return cache[key]

            # Check if someone is computing this result
            with cache_lock:
                # Double-check cache while holding lock
                if key in cache:
                    return cache[key]

                # Check if another thread is computing
                if key in computing:
                    # Wait for the other thread to finish
                    event = computing[key]
                else:
                    # We are the first - create event and mark as computing
                    event = threading.Event()
                    computing[key] = event
                    event = None  # Signal that we should compute

            # If event exists, wait for other thread
            if event is not None:
                event.wait()
                return cache[key]

            # We are the chosen one - do the computation
            try:
                result = func(*args, **kwargs)

                # Cache the result
                with cache_lock:
                    cache[key] = result

                return result

            finally:
                # Signal waiting threads and cleanup
                with cache_lock:
                    event = computing.pop(key)
                    event.set()

        # Add cache inspection methods
        wrapper.cache_info = lambda: {
            'size': len(cache),
            'maxsize': maxsize,
            'currsize': len(cache),
        }
        wrapper.cache_clear = cache.clear

        return wrapper

    return decorator


def shared_process_cache_factory():
    """
    Factory to create a shared process cache decorator with its own Manager.

    The Manager is created lazily on first use, making this safe for Windows spawn mode.
    When a module is imported in spawn mode, the Manager won't be created until
    the decorated function is actually called.

    Returns:
        tuple: (decorator, manager_ref) where decorator can be used as @decorator
        and manager_ref is a dict with 'manager' key (access as manager_ref['manager'])

    Example:
        >>> decorator, manager_ref = shared_process_cache_factory()
        >>> @decorator
        ... def multiply(x, y):
        ...     return x * y
        >>> result = multiply(3, 4)  # Manager created here, on first call
        >>> result
        12
        >>> multiply(3, 4)  # Cached
        12
        >>> info = multiply.cache_info()
        >>> info['size']
        1
        >>> info['type']
        'shared_process'
        >>> multiply.cache_clear()
        >>> multiply.cache_info()['size']
        0
        >>> manager_ref['manager'].shutdown()

    Process-pool safe: Yes (cache shared across all processes)
    Thread-pool safe: Yes (uses process-wide lock)
    """
    # Lazy initialization - Manager created on first function call
    _manager_state = {'manager': None, 'dict': None, 'lock': None}

    def _ensure_manager():
        """Ensure Manager is created (idempotent)"""
        if _manager_state['manager'] is None:
            _manager_state['manager'] = Manager()
            _manager_state['dict'] = _manager_state['manager'].dict()
            _manager_state['lock'] = _manager_state['manager'].Lock()

    def decorator(func: Callable) -> Callable:
        @functools.wraps(func)
        def wrapper(*args, **kwargs):
            # Lazy init: Create Manager on first call, not at import time
            _ensure_manager()

            # Get manager objects (DictProxy and Lock, not plain dict/Lock)
            manager_dict = _manager_state['dict']
            manager_lock = _manager_state['lock']

            # Create cache key (must be picklable for IPC)
            key = (func.__name__, args, tuple(sorted(kwargs.items())))

            # ATOMIC: Check cache and compute under lock
            # pylint: disable=not-context-manager,unsupported-membership-test,unsubscriptable-object,unsupported-assignment-operation
            with manager_lock:
                # Check if already in cache
                if key in manager_dict:
                    return manager_dict[key]

                # Not in cache - compute it (under lock)
                result = func(*args, **kwargs)

                # Store in shared cache
                manager_dict[key] = result

                return result

        # Add cache inspection methods
        def cache_info():
            _ensure_manager()
            return {
                'size': len(_manager_state['dict']),
                'type': 'shared_process',
            }

        def cache_clear():
            _ensure_manager()
            _manager_state['dict'].clear()

        wrapper.cache_info = cache_info
        wrapper.cache_clear = cache_clear

        return wrapper

    # Return decorator and a reference to get the manager (for cleanup)
    return decorator, _manager_state


def shared_process_cache(manager_dict, manager_lock):
    """
    Cross-process shared cache decorator using multiprocessing.Manager.

    This decorator creates a cache that is shared across ALL processes.
    When multiple processes call the decorated function with the same
    arguments, only ONE process computes the result, and all others
    get the cached value from shared memory.

    Args:
        manager_dict: multiprocessing.Manager().dict() - shared cache storage
        manager_lock: multiprocessing.Manager().Lock() - shared lock

    Example:
        >>> from multiprocessing import Manager
        >>> manager = Manager()
        >>> cache_dict = manager.dict()
        >>> cache_lock = manager.Lock()
        >>> @shared_process_cache(cache_dict, cache_lock)
        ... def add(a, b):
        ...     return a + b
        >>> result = add(10, 20)
        >>> result
        30
        >>> add(10, 20)  # Cached
        30
        >>> len(cache_dict)
        1
        >>> cache_dict.clear()
        >>> manager.shutdown()

    Note: This has IPC (Inter-Process Communication) overhead, so it's
    only beneficial for expensive computations that are frequently reused
    across processes.

    Process-pool safe: Yes (cache shared across all processes)
    Thread-pool safe: Yes (uses process-wide lock)
    """
    def decorator(func: Callable) -> Callable:
        @functools.wraps(func)
        def wrapper(*args, **kwargs):
            # Create cache key (must be picklable for IPC)
            key = (func.__name__, args, tuple(sorted(kwargs.items())))

            # ATOMIC: Check cache and compute under lock
            # Note: We hold the lock during computation to prevent duplicates
            # For more sophisticated approach, could use Manager().Event() like synchronized_cache
            with manager_lock:
                # Check if already in cache
                if key in manager_dict:
                    return manager_dict[key]

                # Not in cache - compute it (under lock)
                result = func(*args, **kwargs)

                # Store in shared cache
                manager_dict[key] = result

                return result

        # Add cache inspection methods
        wrapper.cache_info = lambda: {
            'size': len(manager_dict),
            'type': 'shared_process',
        }
        wrapper.cache_clear = manager_dict.clear

        return wrapper

    return decorator


# Convenience function to create a shared cache setup
def create_shared_cache():
    """
    Create Manager objects for shared process cache.

    Returns:
        tuple: (manager, shared_cache, shared_lock) for use with @shared_process_cache

    Example:
        >>> manager, cache, lock = create_shared_cache()
        >>> @shared_process_cache(cache, lock)
        ... def square(x):
        ...     return x * x
        >>> result = square(5)
        >>> result
        25
        >>> square(5)  # Cached
        25
        >>> len(cache)
        1
        >>> manager.shutdown()
    """
    manager = Manager()
    return manager, manager.dict(), manager.Lock()
