"""
Common buffer allocation strategies for AIE4 tilers
"""

from typing import Optional, List
from tiler.base_tiler import HW_CONFIG
from tiler.base_l1_buffer_allocator import BaseL1BufferAllocator
from tiler.buffer_types import BufferSpec, AllocationResult, BufferPair


class L1BufferAllocator(BaseL1BufferAllocator):
    """
    Generic L1 buffer allocator with multiple strategies.
    Supports bank splitting, double buffering, and knapsack optimization.
    """

    def __init__(
        self,
        core_stack_addr: int = None,
        core_bank_size: int = HW_CONFIG.CORE_BANK_SIZE,
        memory_align: int = HW_CONFIG.MEMORY_ALIGNMENT,
        use_software_banks: bool = True,
    ):
        super().__init__(core_stack_addr, core_bank_size, memory_align, use_software_banks)

    def allocate_simple(
        self, buffers: List[BufferSpec]
    ) -> Optional[AllocationResult]:
        """
        Simple allocation without bank splitting.
        Tries double buffering first, then falls back to single buffering.
        """
        # Try double buffering
        result = self._try_sequential_allocation(buffers, double_buffer=True)
        if result:
            return AllocationResult(allocations=result)

        # Fall back to single buffering
        result = self._try_sequential_allocation(buffers, double_buffer=False)
        return AllocationResult(allocations=result) if result else None

    def allocate_cpsat(
        self, buffers: List[BufferSpec], exclusivity_pairs: Optional[List[BufferPair]] = None
    ) -> Optional[AllocationResult]:
        """Allocation using optimal constrained satisfaction formulation."""
        # Calculate ping size
        ping_size = sum(b.size for b in buffers if b.is_ping)
        if ping_size > self.core_stack_addr:
            return None

        # Use CP-SAT
        result = self._try_cpsat_allocation_v2(buffers, exclusivity_pairs=exclusivity_pairs)
        return AllocationResult(allocations=result) if result else None
