"""
Multi-bin packing with constraints using OR-Tools CP-SAT

Features:
- Multiple bins (software banks) with same capacity
- Mandatory items (must be placed)
- Optional items (maximize placement)
- Alignment constraints (power of 2)
- Exclusivity constraints (items that cannot share a bin)
- Items cannot span multiple bins
"""

from dataclasses import dataclass
from typing import List, Tuple, Optional, NamedTuple
import os
from ortools.sat.python import cp_model
from tiler.buffer_types import BinItem
from tiler.base_tiler import get_os_core_count
from graph.utilities import logger


class BinAddr(NamedTuple):
    """Address of an item in a bin"""
    item: BinItem
    offset: int


class BinContent(NamedTuple):
    """Contents of a single bin"""
    bin_id: int
    items: List[BinAddr]


@dataclass
class PackingResult:
    """Result of bin packing"""
    bins: List[BinContent]
    num_bins_used: int
    num_items_placed: int
    total_waste: int

    def print_summary(self):
        """Print packing summary"""
        logger.debug("\n{'='*60}")
        logger.debug("Bins used: %s", self.num_bins_used)
        logger.debug("Items placed: %s", self.num_items_placed)
        logger.debug("Total waste: %s", self.total_waste)
        logger.debug("{'='*60}\n")

        for bin_content in sorted(self.bins, key=lambda x: x.bin_id):
            bin_id, placements = bin_content
            if not placements:
                continue

            logger.debug("Bin %s:", bin_id)
            for item, offset in sorted(placements, key=lambda x: x.offset):
                end = offset + item.size
                req = "✓ REQUIRED" if item.must_place else "  optional"
                logger.debug("  [%4d-%4d] %-15s size=%4d align=%3d [%s]",
                             offset, end-1, item.name, item.size, item.alignment, req)
            logger.debug("")


class BinPacker:
    """Multi-bin packing solver with constraints"""

    def __init__(
        self,
        bin_capacity: int,
        num_bins: int,
        items: List[BinItem],
        exclusivity_pairs: List[Tuple[str, str]] = None,
        time_limit_seconds: float = 30.0
    ):
        """
        Args:
            bin_capacity: Capacity of each bin
            num_bins: Maximum number of bins available
            items: List of items to pack
            exclusivity_pairs: Pairs of item names that cannot share a bin
            time_limit_seconds: Solver time limit
        """
        self.bin_capacity = bin_capacity
        self.num_bins = num_bins
        self.items = items
        self.exclusivity_pairs = exclusivity_pairs or []
        self.time_limit = time_limit_seconds

        # Build item name to index mapping
        self.item_index = {item.name: i for i, item in enumerate(items)}

        logger.debug("Bin capacity: %s, Num bins: %s", bin_capacity, num_bins)
        logger.debug("Total items: %s", len(items))
        logger.debug("Exclusivity pairs: %s", self.exclusivity_pairs)
        for item in items:
            logger.debug("  Item: %s, Size: %s, Align: %s, Priority: %s, Must place: %s",
                         item.name, item.size, item.alignment, item.priority, item.must_place)

    def solve(self) -> Optional[PackingResult]:
        """Solve the bin packing problem"""
        n = len(self.items)

        # Handle empty items case
        if n == 0:
            return PackingResult(
                bins=[BinContent(bin_id=b, items=[]) for b in range(self.num_bins)],
                num_bins_used=0,
                num_items_placed=0,
                total_waste=0
            )

        model = cp_model.CpModel()

        # Variables
        # placed[i] = 1 if item i is placed
        placed = [model.new_bool_var(f'placed_{i}') for i in range(n)]

        # bin_of[i] = which bin item i is in (-1 if not placed)
        bin_of = [model.new_int_var(-1, self.num_bins - 1, f'bin_{i}') for i in range(n)]

        # Start position of each item within its bin
        start = []
        for i in range(n):
            item = self.items[i]
            align = item.alignment
            max_start = self.bin_capacity - item.size

            # Create domain with only aligned positions
            if align > 1:
                possible_starts = list(range(0, max_start + 1, align))
                if not possible_starts:
                    possible_starts = [0]
                start_var = model.new_int_var_from_domain(
                    cp_model.Domain.from_values(possible_starts),
                    f'start_{i}'
                )
            else:
                start_var = model.new_int_var(0, max_start, f'start_{i}')

            start.append(start_var)

        # End position = start + size
        end = [model.new_int_var(0, self.bin_capacity, f'end_{i}') for i in range(n)]

        # bin_used[b] = 1 if bin b contains at least one item
        bin_used = [model.new_bool_var(f'bin_used_{b}') for b in range(self.num_bins)]

        # Constraints

        # 1. MANDATORY ITEMS: must be placed
        for i in range(n):
            if self.items[i].must_place:
                model.add(placed[i] == 1)

        # 2. If placed, bin_of >= 0; if not placed, bin_of == -1
        for i in range(n):
            model.add(bin_of[i] >= 0).only_enforce_if(placed[i])
            model.add(bin_of[i] == -1).only_enforce_if(placed[i].Not())

        # 3. Size constraint: end = start + size
        for i in range(n):
            model.add(end[i] == start[i] + self.items[i].size)

        # 4. Capacity constraint (only for placed items)
        for i in range(n):
            model.add(end[i] <= self.bin_capacity).only_enforce_if(placed[i])

        # 5. NO OVERLAP within same bin
        for i in range(n):
            for j in range(i + 1, n):
                # Both items are placed
                both_placed = model.new_bool_var(f'both_placed_{i}_{j}')
                model.add(placed[i] + placed[j] == 2).only_enforce_if(both_placed)
                model.add(placed[i] + placed[j] < 2).only_enforce_if(both_placed.Not())

                # Items are in same bin
                same_bin = model.new_bool_var(f'same_bin_{i}_{j}')
                model.add(bin_of[i] == bin_of[j]).only_enforce_if([both_placed, same_bin])
                model.add(bin_of[i] != bin_of[j]).only_enforce_if([both_placed, same_bin.Not()])

                # If both placed and in same bin, enforce no overlap
                # Either i ends before j starts, OR j ends before i starts
                i_before_j = model.new_bool_var(f'{i}_before_{j}')

                model.add(end[i] <= start[j]).only_enforce_if([both_placed, same_bin, i_before_j])
                model.add(end[j] <= start[i]).only_enforce_if([both_placed, same_bin, i_before_j.Not()])

                # Must choose one ordering if in same bin
                # This is implicitly enforced by the above constraints

        # 6. EXCLUSIVITY CONSTRAINT: certain pairs cannot share a bin
        for name1, name2 in self.exclusivity_pairs:
            if name1 not in self.item_index or name2 not in self.item_index:
                continue

            i = self.item_index[name1]
            j = self.item_index[name2]

            # If both placed, they must be in different bins
            both_placed = model.new_bool_var(f'excl_both_placed_{i}_{j}')
            model.add(placed[i] + placed[j] == 2).only_enforce_if(both_placed)
            model.add(placed[i] + placed[j] < 2).only_enforce_if(both_placed.Not())

            model.add(bin_of[i] != bin_of[j]).only_enforce_if(both_placed)

        # 7. Link bin_used with placements
        for b in range(self.num_bins):
            items_in_bin = [model.new_bool_var(f'item_{i}_in_bin_{b}') for i in range(n)]

            for i in range(n):
                model.add(bin_of[i] == b).only_enforce_if(items_in_bin[i])
                model.add(bin_of[i] != b).only_enforce_if(items_in_bin[i].Not())

            # bin_used[b] = 1 if any item is in bin b
            model.add_max_equality(bin_used[b], items_in_bin)

        # 8. Symmetry breaking: use bins in order (bin i not used => bin i+1 not used)
        for b in range(self.num_bins - 1):
            model.add(bin_used[b + 1] <= bin_used[b])

        # Objective: Maximize items placed (primary), minimize bins used (secondary)
        # Use large weight for placement to ensure it dominates
        placement_weight = 10000
        total_value = sum(placed[i] * self.items[i].priority for i in range(n))
        bins_used_count = sum(bin_used[b] for b in range(self.num_bins))

        # Maximize placement value, minimize bins used
        # Note: We negate to maximize (since CP-SAT minimizes)
        model.maximize(
            total_value * placement_weight - bins_used_count
        )

        # Solve
        solver = cp_model.CpSolver()
        solver.parameters.max_time_in_seconds = self.time_limit
        solver.parameters.log_search_progress = False

        # Use deterministic settings for pytest (env var PYTEST_CURRENT_TEST is set by pytest)
        # For production: use parallel search for better performance
        is_pytest = 'PYTEST_CURRENT_TEST' in os.environ
        if is_pytest:
            solver.parameters.num_workers = 1
            solver.parameters.random_seed = 42
        else:
            solver.parameters.num_workers = get_os_core_count()
            # Don't set random_seed for production - let solver explore freely

        logger.debug("CP-SAT solver config: num_workers=%s, random_seed=%s, is_pytest=%s",
                     solver.parameters.num_workers,
                     solver.parameters.random_seed if is_pytest else 'default',
                     is_pytest)

        status = solver.solve(model)

        if status not in [cp_model.OPTIMAL, cp_model.FEASIBLE]:
            logger.debug("Solver status: %s", solver.status_name(status))
            return None

        # Extract solution
        bins_list = [BinContent(bin_id=b, items=[]) for b in range(self.num_bins)]

        for i in range(n):
            if solver.value(placed[i]):
                bin_id = solver.value(bin_of[i])
                offset = solver.value(start[i])
                bins_list[bin_id].items.append(BinAddr(self.items[i], offset))

        # Calculate waste
        total_waste = 0
        for bin_id, placements in bins_list:
            if not placements:
                continue

            # Sort by offset
            placements.sort(key=lambda x: x[1])

            # Initial gap
            total_waste += placements[0][1]

            # Gaps between items
            for j in range(len(placements) - 1):
                gap = placements[j + 1][1] - (placements[j][1] + placements[j][0].size)
                total_waste += gap

        num_bins_used = sum(solver.value(bin_used[b]) for b in range(self.num_bins))
        num_placed = sum(solver.value(placed[i]) for i in range(n))

        return PackingResult(
            bins=bins_list,
            num_bins_used=num_bins_used,
            num_items_placed=num_placed,
            total_waste=total_waste
        )
