''' Arg Parser for Build System '''
from __future__ import annotations
import argparse
import json
import os
import sys
import yaml


def parse_args_with_cfg(parser: argparse.ArgumentParser, argv=None) -> argparse.Namespace:
    """Probe ONLY for config flag (don't parse full CLI yet)"""
    probe = argparse.ArgumentParser(add_help=False)
    probe.add_argument("--cfg", "--config", "-c", "-cfg", dest="cfg")
    cfg_ns, _ = probe.parse_known_args(argv)

    # Load config if provided
    if cfg_ns.cfg:
        path = cfg_ns.cfg
        if not os.path.isfile(path):
            raise SystemExit(f"[ERR] Config file not found: {path}")

        with open(path, "r", encoding="utf-8") as f:
            if path.lower().endswith((".yml", ".yaml")):
                data = yaml.safe_load(f) or {}
            else:
                data = json.load(f)

        if not isinstance(data, dict):
            raise SystemExit(f"[ERR] Config root must be a dictionary: {path}")

        # Normalize list -> CSV strings for flags your CLI expects as strings
        for k in ("layer_ids", "ifm_path"):
            v = data.get(k)
            if isinstance(v, list):
                data[k] = ",".join(str(x) for x in v)

        # Filter unknown keys (helps catch mistakes)
        # pylint: disable=protected-access
        known = {a.dest for a in parser._actions if getattr(a, "dest", None)}
        unknown = [k for k in data if k not in known]
        if unknown:
            print(f"[WARN] Ignoring unknown config keys: {', '.join(unknown)}", file=sys.stderr)

        # Enforce mutually exclusive args at YAML config level
        if "skip_op" in data and "include_op" in data:
            if data.get("skip_op") is not None and data.get("include_op") is not None:
                raise RuntimeError(
                    "[ERR] Config file cannot specify both 'skip_op' and 'include_op'. "
                    "They are mutually exclusive."
                )

        # Apply config as defaults
        parser.set_defaults(**{k: v for k, v in data.items() if k in known})

    # Final parse: CLI overrides config
    return parser.parse_args(argv)


def _str2bool(x: str) -> bool:
    ''' Handle CLI boolean. '''
    return str(x).lower() in ("true", "1", "yes", "y", "t")


def make_build_parser() -> argparse.ArgumentParser:
    """ Factory for the build_aie4.py CLI parser. """
    parser = argparse.ArgumentParser(
        description="Build ONNX model for AIE4.",
        formatter_class=argparse.RawTextHelpFormatter
    )

    # Single Operator Compilation
    parser.add_argument("-pdi", "--build_pdi", type=_str2bool, default=True,
                        help="Build standalone PDI for operator testing. (Default -> True)")
    parser.add_argument("--chain_di", type=_str2bool, default=False,
                        help="Build Chained DI for model testing. (Default -> False)")

    # Fused L2/L3 Allocation JSON Compilation
    parser.add_argument("--json", type=str, help="Path to a JSON file containing build configuration.")
    parser.add_argument("--layer_ids", type=str, help="(Optional) Key of block in JSON to compile. Compiles all blocks if not set.")
    parser.add_argument("--ifm_path", type=str, help="(Optional) Read ifm bin file path(s) for the FIRST block.")
    parser.add_argument("--wgt_path", type=str, help="(Optional) Read wgt bin file path for the FIRST block.")
    parser.add_argument("-i", "--build_sim", type=str, help="Set the path of input bin directory. (Default -> 'Output/fused_hw_package')")

    # Fused ONNX model Compilation
    parser.add_argument("-um", "--unfused_model", type=str, default="", help="Path to a un-fused ONNX model.")
    parser.add_argument("-fm", "--fused_model", type=str, help="Path to a Fused ONNX model.")
    parser.add_argument("-ir", "--ir_json", type=str, help="Path to Fused IR JSON.")
    parser.add_argument("-mdp", "--model_data_path", type=str, default="", help="Path to Model Data.")
    parser.add_argument("-rmd", "--read_model_data", type=_str2bool, default=False, help="Read Model Data?")
    parser.add_argument("-un", "--unique_nodes", type=str, help="Path to Unique Nodes JSON.")
    parser.add_argument("-tm", "--tensor_map", type=str, default="", help="Path to Tensor Map JSON.")
    parser.add_argument("-skip", "--skip", action="store_true", help="Only generate alloc.json. No Layer/Subgraph compilation.")
    parser.add_argument("-L2L3", "--both_L2L3", action="store_true", help="Allocate both on L2 and L3")
    parser.add_argument("-node", "--node_list", type=str, default=None, help="Path to Node List for ONNX Graph Partitioner.")
    parser.add_argument("--cfg", "-cfg", "--config", help="Path to YAML/JSON build config.")
    parser.add_argument("-nodes", "--gen_node_list", type=bool, default=False, help="Only generate nodelist for all subgraphs.")
    parser.add_argument("-cut_graphs", "--save_cut_graphs", type=bool, default=False, help="Save all subgraphs generated by Partitioner in ONNX format.")
    parser.add_argument("-ca", "--cont_alloc", type=bool, default=False, help="Force continuous allocation.")

    # QDQ Data Type Mode
    parser.add_argument("-fp16", "--is_qdq_fp16", type=_str2bool, default=True, help="QDQ datatpye is FP16 or BF16? (Default -> False (QDQ DType is BF16))")

    # Mutually exclusive operator filters
    op_filter_group = parser.add_mutually_exclusive_group()
    op_filter_group.add_argument("-skip_op", "--skip_op", type=str, default=None, help="Commad Separated List of Operators that should be skipper while compiling.")
    op_filter_group.add_argument("-include_op", "--include_op", type=str, default=None, help="Commad Separated List of Operators that should be included while compiling.")

    # Num Parallel Workers
    parser.add_argument("-workers", "--num_workers", type=int, default=None, help="Number of workers for parallel subgraph compilation.")

    # Backend mode
    parser.add_argument("-t", "--target", type=str,
                        choices=["dataflow", "sim", "cert_sim", "cert"],
                        default="dataflow",
                        help="Backend mode: dataflow = dmacompiler-codegen, sim = Adf, cert = CERT.")

    # Output folder
    parser.add_argument("-o", "--outfolder", type=str, default="Output",
                        help="Set the path of Output directory. (Default -> 'Output')")

    # Cleanup flag
    parser.add_argument("-clean", "--clean", action="store_true",
                        help="If set, removes the Output folder and its contents before processing.")

    # Enable Logger
    parser.add_argument("--enable_log", type=_str2bool, default=False,
                        help="Enable or disable logging (True/False)")

    # ML Timeline
    parser.add_argument("-ml_timeline", "--ml_timeline", type=_str2bool, default=False, help="Enable ML Timeline Profiling.")

    # Device generation
    parser.add_argument("--device", type=str, choices=["mds", "swv"], default="mds",
                        help="Device mode: Default is `mds`. `mds` refers to Medusa; `swv` refers to SoundWave.")

    return parser


def parse_build_args(argv: list[str] | None = None):
    """ Convenience wrapper for tests and scripts. """
    return make_build_parser().parse_args(argv)
