"""Script to generate max BO Size for all subgraphs"""
import json
import os
import threading
from typing import List, Dict, Any, Tuple, Optional
from threading import Lock


def build_subgraph_report_async(
    allocator_json_path: str,
    alloc_json_inmem: dict[str, Any],
    subgraphs: Dict[int, list[int]],
    subgraph_name_map: Dict[int, str],
    verbose: bool = False,
):
    """
    Asynchronously:
      1) Loads allocator JSON from `allocator_json_path`
      2) For each subgraph (list of node ids), computes BO_0 and BO_1 sizes:
         - For each node's L3 dict, look at each value (expected [bo, offset, size])
         - For each BO (0 or 1), compute max(offset + size) across ALL entries in the subgraph
      3) Updates context.json (located at <dir-of-alloc>/cut_graphs/context.json) by subgraph name:
         - context[subgraph_name]["subgraph_index"] = index
         - context[subgraph_name]["bo_sizes"] = {"BO_0": x, "BO_1": y}
    This function launches a background thread and returns immediately so your main code continues.
    """

    # Resolve paths and output targets in the same directory tree as allocator_json_path
    alloc_dir = os.path.dirname(os.path.abspath(allocator_json_path))
    context_json_path = os.path.join(alloc_dir, "cut_graphs", "context.json")

    _print_lock = Lock()  # keep logs tidy if verbose=True

    def _vprint(*args, **kwargs):
        if verbose:
            with _print_lock:
                print(*args, **kwargs)

    def _load_json(path: str) -> Any:
        """Load JSON file and return the parsed object."""
        with open(path, "r", encoding="utf-8") as f:
            return json.load(f)

    def _dump_json(obj: Any, path: str):
        """Write JSON with indentation, creating parent dirs if needed."""
        os.makedirs(os.path.dirname(path) or ".", exist_ok=True)
        with open(path, "w", encoding="utf-8") as f:
            json.dump(obj, f, indent=2)

    def _get_node_block(alloc_json: Dict[str, Any], node_id: int) -> Optional[Dict[str, Any]]:
        """Handle string or int node keys gracefully."""
        return alloc_json.get(str(node_id)) or alloc_json.get(node_id)

    def _safe_iter_L3_entries(l3_dict: Any):
        """
        Yields (bo, offset, size) triples from an L3 dict where values look like [bo, offset, size].
        Ignores malformed entries.
        """
        if not isinstance(l3_dict, dict):
            return
        for _, val in l3_dict.items():
            # Some pipelines may hand tuples or lists; normalize to list access.
            if (isinstance(val, (list, tuple)) and len(val) >= 3 and isinstance(val[0], int) and isinstance(val[1], int) and isinstance(val[2], int)):
                bo, offset, size = int(val[0]), int(val[1]), int(val[2])
                yield bo, offset, size

    def _compute_bo_sizes_for_nodes(nodes: List[int], alloc_json: Dict[str, Any]) -> Tuple[int, int, List[str]]:
        """
        Returns (bo0_size, bo1_size) where each is max(offset+size) across the subgraph.
        Defaults to 0 if nothing found.
        """
        bo_max = {0: 0, 1: 0}
        nodelist = []
        for nid in nodes:
            blk = _get_node_block(alloc_json, nid)
            if not blk:
                _vprint(f"[warn] Node {nid} not found in allocator JSON.")
                continue
            l3 = blk.get("L3")
            nodelist.append(blk.get("name", "Layer Name Not Found!"))
            if not l3:
                # Gracefully skip nodes without L3
                continue
            for bo, off, sz in _safe_iter_L3_entries(l3):
                if bo in (0, 1):
                    candidate = off + sz
                    if candidate > bo_max[bo]:
                        bo_max[bo] = candidate
        return bo_max[0], bo_max[1], nodelist

    def _worker():
        """Worker function executed in background thread."""
        # Load allocator json (nodes + L3)
        _vprint("[info] Loading allocator JSON:", allocator_json_path)
        alloc_json = alloc_json_inmem

        # Load or initialize context.json
        _vprint("[info] Loading context JSON:", context_json_path)
        try:
            context_obj = _load_json(context_json_path)
            if not isinstance(context_obj, dict):
                _vprint("[warn] context.json root is not a dict; reinitializing to {}")
                context_obj = {}
        except FileNotFoundError:
            _vprint("[warn] context.json not found; will create a new one.")
            context_obj = {}

        # Process each subgraph, compute bo sizes, and update by subgraph name
        _vprint("[info] Processing subgraphs and updating context...")
        for idx, nodes in subgraphs.items():
            # Map numeric subgraph index to a subgraph_name via provided dict
            subgraph_name = subgraph_name_map.get(idx)
            if not subgraph_name:
                _vprint(f"[warn] No subgraph name provided for index {idx}; skipping.")
                continue

            bo0, bo1, nodelist = _compute_bo_sizes_for_nodes(nodes, alloc_json)

            # Ensure the entry exists and is a dict; context[subgraph_name] holds misc info
            entry = context_obj.get(subgraph_name)
            if entry is None:
                _vprint(f"[warn] subgraph_name '{subgraph_name}' not found in context.json. Creating new entry.")
                entry = {}
                context_obj[subgraph_name] = entry
            elif not isinstance(entry, dict):
                _vprint(f"[warn] subgraph_name '{subgraph_name}' exists but is not a dict. Normalizing it to dict.")
                entry = {"_raw": entry}
                context_obj[subgraph_name] = entry

            # Update required fields
            entry["subgraph_index"] = idx
            entry["layer_ids"] = nodes
            entry["nodelist"] = nodelist
            entry["bo_sizes"] = {"BO_0": bo0, "BO_1": bo1}

        # Write back context.json in place
        _dump_json(context_obj, context_json_path)
        _vprint(f"[info] Updated context.json → {context_json_path}")

    # Start work in background thread and return immediately
    thread = threading.Thread(target=_worker, daemon=True)
    thread.start()
