"""
summarize_noncompilable_ops.py

Utility to analyze model subgraph JSON summaries and report non-compilable
operators.

The script:
- Reads a JSON file containing per-subgraph metadata, indexed by string keys
- Filters entries where `is_compilable == False`
- Aggregates and counts the frequency of the `op` field
- Produces a human-readable summary table using `tabulate`

This module is designed to be:
- Importable as a library function for build / artifact pipelines
- Executable as a standalone CLI tool via the `--json` argument

Typical usage (CLI):
    python summarize_noncompilable_ops.py --json subgraph_summary.json

Typical usage (import):
    from summarize_noncompilable_ops import summarize_noncompilable_ops
    output = summarize_noncompilable_ops(json_path)
    Path("summary.log").write_text(output)
"""

from __future__ import annotations

import argparse
import json
from collections import Counter
from pathlib import Path
from typing import Any

from tabulate import tabulate


def summarize_noncompilable_ops(json_path: str | Path) -> str:
    """
    Generate a formatted summary table of non-compilable operators.

    The function scans the provided JSON file, identifies all blocks where
    `is_compilable` is explicitly set to False, extracts the corresponding
    `op` field, and computes the frequency of each operator.

    Args:
        json_path:
            Path to the JSON file containing alloctor output.

    Returns:
        A string containing a formatted table with two columns:
        - `op`: Operator name
        - `frequency`: Number of occurrences where the operator is non-compilable

    Raises:
        FileNotFoundError:
            If the JSON file does not exist.
        ValueError:
            If the JSON content is not a dictionary at the top level.
        json.JSONDecodeError:
            If the file contents are not valid JSON.
    """
    path = Path(json_path)
    if not path.exists():
        raise FileNotFoundError(f"JSON not found: {path}")

    with path.open("r", encoding="utf-8") as f:
        data: Any = json.load(f)

    if not isinstance(data, dict):
        raise ValueError("Expected top-level JSON object (dict).")

    counter: Counter[str] = Counter()
    op_to_block_ids: dict[str, list[str]] = {}

    for block_id, block in data.items():
        if not isinstance(block, dict):
            continue

        if block.get("is_compilable") is False:
            op = block.get("op")
            op_name = op if isinstance(op, str) else "<missing_op>"

            counter[op_name] += 1
            op_to_block_ids.setdefault(op_name, []).append(str(block_id))

    rows = sorted(counter.items(), key=lambda x: (-x[1], x[0]))

    table_rows = [
        [op_name, freq, ", ".join(op_to_block_ids.get(op_name, []))]
        for op_name, freq in rows
    ]

    return tabulate(
        table_rows,
        headers=["Operator", "Frequency", "Layer ID"],
        tablefmt="github",
    )


def main() -> None:
    """
    Command-line entry point.
    """
    parser = argparse.ArgumentParser(
        description="Summarize non-compilable operators from a JSON file."
    )
    parser.add_argument(
        "--json",
        required=True,
        help="Path to the JSON file to analyze.",
    )
    args = parser.parse_args()

    print(summarize_noncompilable_ops(args.json))


if __name__ == "__main__":
    main()
