import re
import sys
import subprocess
import os

def parse_gpp_command(gpp_command):
    """Parses a g++ command and generates a CMakeLists.txt file."""
    cmake_content = [
        "cmake_minimum_required(VERSION 3.14)",
        "project(txn_dma CXX)",
        "set(CMAKE_CXX_STANDARD 20)",
        "set(CMAKE_CXX_STANDARD_REQUIRED YES)",
        #'set(CMAKE_CXX_FLAGS "${CMAKE_CXX_FLAGS} /EHsc")'  # Corrected quotes
        "if (MSVC)",
        "add_compile_options(/EHsc /wd4710 /wd4711)",
        "endif()"
    ]

    # Extract compiler flags
    flags = re.findall(r"-(Wall|Wextra|w)", gpp_command)
    cmake_flags = " ".join(["-" + flag for flag in flags])
    cmake_content.append(f'set(CMAKE_CXX_FLAGS "{cmake_flags}")')

    # Extract macro definitions including comma-containing values
    definitions = re.findall(r"-D([a-zA-Z_0-9]+(?:=[^ \t\r\n]+)?)", gpp_command)
    cmake_definitions = "\n".join([f"add_definitions(-D{macro.strip()})" for macro in definitions])
    cmake_content.append(cmake_definitions)

    # Extract include directories (-I flags)
    includes = re.findall(r"(?:^|\s)-I\s*(?:\"([^\"]+)\"|([^\s\"]+))", gpp_command)
    cleaned_includes = [inc.replace("\\", "/") for pair in includes for inc in pair if inc]
    cmake_includes = "\n".join([f'include_directories("{inc}")' for inc in cleaned_includes])
    cmake_content.append(cmake_includes)

    # Extract linked libraries and directories (-L flags)
    lib_dirs = re.findall(r"-L\s*\"?([^\s\"]+)\"?", gpp_command)
    cmake_lib_dirs = "\n".join([f'link_directories("{lib.replace(chr(92), "/")}")' for lib in lib_dirs])
    cmake_content.append(cmake_lib_dirs)

    libraries = re.findall(r"(?<=\s)-l([a-zA-Z0-9_]+)", gpp_command)
    cmake_libraries = "\n".join([f'target_link_libraries(txn_dma PRIVATE {lib})' for lib in libraries])

    # Extract source files (.cpp files)
    sources = re.findall(r"[^\s\"]+\.cpp", gpp_command)
    sources_list = "\n".join([f'    "{os.path.normpath(src).replace(chr(92), "/")}"' for src in sources])
    cmake_content.append(f"set(SOURCES\n{sources_list}\n)")
    cmake_content.append("add_executable(txn_dma ${SOURCES})")
    cmake_content.append(cmake_libraries)

    # Write to CMakeLists.txt
    with open("CMakeLists.txt", "w") as cmake_file:
        cmake_file.write("\n".join(cmake_content))

    print("CMakeLists.txt generated successfully!")

def build_and_run():
    """Runs CMake and builds the project."""
    try:
        print("Running CMake...")
        subprocess.run(["cmake", "-S", ".", "-B", "build", "-G", "Visual Studio 17 2022"], check=True)

        print("Building the project...")
        subprocess.run(["cmake", "--build", "build", "--config", "Release"], check=True)

        print("Build successful!")

    except subprocess.CalledProcessError as e:
        print(f"Error occurred: {e}")
        print("Check build logs for issues.")

def run_executable():
    """Executes the txn_dma.exe file after build."""
    exe_path = os.path.join("build", "Release", "txn_dma.exe")

    if os.path.exists(exe_path):
        print(f"Running {exe_path}...")
        subprocess.run([exe_path], check=True)
    else:
        print(f"Error: {exe_path} not found. Make sure the build was successful.")

if __name__ == "__main__":
    if len(sys.argv) < 2:
        print("Usage: python script.py <g++ command>")
        sys.exit(1)

    gpp_command = " ".join(sys.argv[1:])
    parse_gpp_command(gpp_command)
    build_and_run()
    run_executable()