import json
import os
import shutil
import traceback
from typing import List, Optional, Dict, Any

CURRDIR = os.path.dirname(os.path.abspath(__file__))
import sys, traceback

sys.path.append(os.path.join(CURRDIR, ".."))
sys.path.append(os.path.join(CURRDIR, "..", ".."))
sys.path.append(os.path.join(CURRDIR, "..", "..", "dmacompiler"))
sys.path.append(os.path.join(CURRDIR, "..", "..", "OGOAT", "src", "Tiler"))
sys.path.append(CURRDIR)


from pprint import pprint

from OGOAT.src.Tiler.conv_tiling_opt import ConvTilingOpt
from OGOAT.src.Tiler.device import Device
from OGOAT.src.Tiler.kernel import Kernel
from OGOAT.src.Tiler.layer import Layer
from OGOAT.src.Tiler.overlay import Overlay
from conv_common import ConvPingPong, PingPong
from conv_common import ConvDims, iceil, conv_preproc_directives
import conv_y8xc_dataflow_matadd
import conv_L3_dataflow
from dataflow_common import clean_overlay, build_sim_overlay
from dmacompiler import BackEnd
from OGOAT.src.L1_fusion.kernel_func_list import kernel_func_list
from config_loader import waic_config

# Check if GCC is available
gcc_available = shutil.which("gcc") is not None


def prepare_compile_flags(layer: Layer, kernel: Kernel, back_end: BackEnd) -> List[str]:
    """
    Prepare compile flags based on layer and kernel properties.

    Args:
        layer: Layer object
        kernel: Kernel object

    Returns:
        List of compile flags
    """
    (Py_b, Px_b, Py_a, Px_a) = layer.padding
    op_type = layer.op_type
    enable_add = kernel.enable_add

    if back_end == BackEnd.Adf:
        compile_flags = [
            f'--Xpreproc="-DPAD_Y_BEFORE={Py_b}"',
            f'--Xpreproc="-DPAD_Y_AFTER={Py_a}"',
            f'--Xpreproc="-DPAD_X_BEFORE={Px_b}"',
            f'--Xpreproc="-DPAD_X_AFTER={Px_a}"',
            f'--Xpreproc="-DENABLE_ADD={int(enable_add)}"',
            f'--Xpreproc="-DIS_XINT8={kernel.is_xint8}"',
            f'--Xpreproc="-DIS_A8W8={kernel.is_a8w8}"',
        ]
    else:
        compile_flags = [
            f"-DPAD_Y_BEFORE={Py_b}",
            f"-DPAD_Y_AFTER={Py_a}",
            f"-DPAD_X_BEFORE={Px_b}",
            f"-DPAD_X_AFTER={Px_a}",
            f"-DENABLE_ADD={int(enable_add)}",
            f"-DIS_XINT8={kernel.is_xint8}",
            f"-DIS_A8W8={kernel.is_a8w8}",
        ]

    if "leakyrelu" in op_type:
        print("Given op is fused with leakyrelu")
        compile_flags += (
            ['--Xpreproc="-DFUSED_OP=LeakyRelu"']
            if back_end == BackEnd.Adf
            else ["-DFUSED_OP=LeakyRelu"]
        )
    elif "relu" in op_type:
        print("Given op is fused with relu")
        compile_flags += (
            ['--Xpreproc="-DFUSED_OP=Relu"']
            if back_end == BackEnd.Adf
            else ["-DFUSED_OP=Relu"]
        )
    else:
        print("Conv is not fused")
        compile_flags += (
            ['--Xpreproc="-DFUSED_OP=None"']
            if back_end == BackEnd.Adf
            else ["-DFUSED_OP=None"]
        )

    return compile_flags


def extract_tiling_params(
    tiling_params: Dict[str, Any], config_idx: int
) -> Dict[str, Any]:
    """Extract tiling parameters for a specific configuration index

    Args:
        tiling_params: Original tiling parameters dictionary
        config_idx: Index of the configuration to extract

    Returns:
        Dict[str, Any]: Extracted tiling parameters for the specific configuration
    """
    # Create a copy of the original tiling params
    params_copy = {}
    for key, value in tiling_params.items():
        if isinstance(value, dict):
            params_copy[key] = value.copy()
        else:
            params_copy[key] = value

    # Update the core_tile_params with the specific configuration
    for field in ["subvols", "iters", "L1_sizes_addrs", "ping_pong"]:
        if (
            field in params_copy["core_tile_params"]
            and len(params_copy["core_tile_params"][field]) > config_idx
        ):
            params_copy["core_tile_params"][field] = [
                params_copy["core_tile_params"][field][config_idx]
            ]

    # Update the mem_tile_params with the specific configuration
    for field in ["subvols", "iters", "sizes", "configs"]:
        if (
            field in params_copy["mem_tile_params"]
            and len(params_copy["mem_tile_params"][field]) > config_idx
        ):
            params_copy["mem_tile_params"][field] = [
                params_copy["mem_tile_params"][field][config_idx]
            ]

    if (
        "mode" in params_copy["overlay_info"]
        and len(params_copy["overlay_info"]["mode"]) > config_idx
    ):
        params_copy["overlay_info"]["mode"] = params_copy["overlay_info"]["mode"][
            config_idx
        ]

    return params_copy


def prepare_conv_dims(
    tiling_params: dict,
    layer: Layer,
    kernel: Kernel,
    overlay: Overlay,
    is_standalone_dwc: bool = False,
) -> ConvDims:
    """
    Prepare the ConvDims object from the given parameters.

    Args:
        tiling_params: Tiling parameters from JSON
        layer: Layer object
        kernel: Kernel object
        overlay: Overlay object
        is_standalone_dwc: Whether this is a standalone DWC operation

    Returns:
        ConvDims object
    """
    op_type = layer.op_type

    if "add" in op_type:
        print("Conv is fused with ElwAdd")

    # NOTE: creating bool flag to detect whether the given conv op is L2_fused or not
    enable_L2_IFM = layer.in_act_residency == "L2"
    enable_L2_OFM = layer.out_act_residency == "L2"
    load_input_from_ddr = True
    store_output_to_ddr = True

    # Extract values from layer object
    # (_, Ci, Yi, Xi) = layer.aligned_in_act_shape
    (_, Yi, Xi, Ci) = layer.aligned_in_act_shape
    # (_, Co, Yo, Xo) = layer.aligned_out_act_shape
    (_, Yo, Xo, Co) = layer.aligned_out_act_shape
    (Ky, Kx) = layer.kernel_shape
    (Sy, Sx) = layer.strides
    (Py_b, Px_b, Py_a, Px_a) = layer.padding

    # Extract values from the JSON data
    # From core_tile_params.subvols
    Cis = tiling_params["core_tile_params"]["subvols"][0]["Cis"]
    Yis = tiling_params["core_tile_params"]["subvols"][0]["Yis"]
    Xis = tiling_params["core_tile_params"]["subvols"][0]["Xis"]
    Cos = tiling_params["core_tile_params"]["subvols"][0]["Cos"]
    Yos = tiling_params["core_tile_params"]["subvols"][0]["Yos"]
    Xos = tiling_params["core_tile_params"]["subvols"][0]["Xos"]

    # From core_tile_params.iters
    Y_loop = tiling_params["core_tile_params"]["iters"][0]["Y_loop"]
    Co_loop = tiling_params["core_tile_params"]["iters"][0]["Co_loop"]
    Ci_loop = tiling_params["core_tile_params"]["iters"][0]["Ci_loop"]
    X_loop = tiling_params["core_tile_params"]["iters"][0]["X_loop"]

    # ifm_streaming_mode = tiling_params['core_tile_params']['subvols'][0]['ifm_streaming_mode']

    conv_pp = ConvPingPong(
        PingPong.from_str(tiling_params["core_tile_params"]["ping_pong"][0]["ifm"]),
        PingPong.from_str(tiling_params["core_tile_params"]["ping_pong"][0]["ofm"]),
        PingPong.from_str(tiling_params["core_tile_params"]["ping_pong"][0]["wgt"]),
        PingPong.from_str(tiling_params["core_tile_params"]["ping_pong"][0]["tdm"]),
    )

    L1_sizes_addrs = tiling_params["core_tile_params"]["L1_sizes_addrs"][0]

    # From overlay object - access core_splits for ofm
    aie_cols, aie_rows = overlay.cols, overlay.rows

    # Extract from core_splits for ofm [Y_split, X_split, Co_split]
    Y_split, X_split, Co_split = map(int, overlay.core_splits["ofm"])

    # Setting bool variable for X8Split
    is_X8_split = X_split == 8
    # Determine bit widths based on data types
    ifm_bytes = layer.in_bytes
    wgt_bytes = layer.wgt_bytes
    ofm_bytes = layer.out_bytes
    bias_bytes = kernel.bias_bytes
    tdm_bytes = kernel.tdm_bytes
    enable_add = kernel.enable_add
    is_xint8 = kernel.is_xint8

    # Get constraints from kernel config
    Y_gran, X_gran, Ci_gran, Co_gran = kernel.subvol_constraints.values()
    X_align = kernel.additional_constraints.get("mem_align", None)

    # Extract values from the JSON data
    # From mem_tile_params.subvols
    mem_tile_params_configs = tiling_params["mem_tile_params"]["configs"][0]
    mem_tile_params_subvols = tiling_params["mem_tile_params"]["subvols"][0]
    mem_tile_params_sizes = tiling_params["mem_tile_params"]["sizes"][0]

    enable_ifm_streaming = mem_tile_params_configs["enable_ifm_streaming"]
    enable_wgt_reuse = mem_tile_params_configs["enable_wgt_reuse"]
    pin_ifm_l1 = mem_tile_params_configs["pin_ifm_l1"]
    pin_wgt_bias_l1 = mem_tile_params_configs["pin_wgt_bias_l1"]

    Com = mem_tile_params_subvols["Com"]
    Xom = mem_tile_params_subvols["Xom"]
    Cim = mem_tile_params_subvols["Cim"]

    wgt_memtile_size = mem_tile_params_sizes["wgt_memtile_size"]
    num_ifm_subv = mem_tile_params_sizes["num_ifm_subv"]
    prm_memtile_size = mem_tile_params_sizes["prm_memtile_size"]
    ifm_memtile_size = mem_tile_params_sizes["ifm_memtile_size"]
    ofm_memtile_size = mem_tile_params_sizes["ofm_memtile_size"]
    conv_kernel_param_size = mem_tile_params_sizes["conv_kernel_param_size"]
    param_subv_size = mem_tile_params_sizes["param_subv_size"]
    mt_co_pack = mem_tile_params_sizes["mt_co_pack"]
    num_pack_wgt_subv = mem_tile_params_sizes["num_pack_wgt_subv"]

    # Preparing the tiling configuration in JSON format
    tiling_json = {
        "input": (Ci, Yi, Xi),
        "output": (Co, Yo, Xo),
        "input_subv": (Cis, Yis, Xis),
        "output_subv": (Cos, Yos, Xos),
        "Y_split": Y_split,
        "X_split": X_split,
        "Co_split": Co_split,
        "is_X8_split": is_X8_split,
        "enable_ifm_streaming": enable_ifm_streaming,
        "original_dimensions": tiling_params["original_dimensions"],
        "host_layer_padding": tiling_params["host_layer_padding"],
        "dma_layer_padding": tiling_params["dma_layer_padding"],
    }
    tiling_json_filename = "tiling.json"
    with open(tiling_json_filename, "w") as f:
        f.write(json.dumps(tiling_json, sort_keys=True, indent=4))

    dims = ConvDims(
        aie_cols,
        aie_rows,
        Ci,
        Cis,
        Ci_gran,
        Co,
        Cos,
        Co_gran,
        Co_split,
        Yi,
        Yis,
        Yo,
        Yos,
        Xi,
        Xis,
        Xo,
        Xos,
        X_gran,
        X_align,
        X_split,
        Y_loop,
        Co_loop,
        Ci_loop,
        X_loop,
        Ky,
        Kx,
        Sy,
        Sx,
        Py_b,
        Px_b,
        Py_a,
        Px_a,
        ifm_bytes * 8,
        wgt_bytes * 8,
        ofm_bytes * 8,
        tdm_bytes * 8,
        L1_sizes_addrs,
        is_standalone_dwc,
        conv_pp=conv_pp,
        enable_L2_IFM=enable_L2_IFM,
        enable_L2_OFM=enable_L2_OFM,
        load_input_from_ddr=load_input_from_ddr,
        store_output_to_ddr=store_output_to_ddr,
        is_X8_split=is_X8_split,
        enable_ifm_streaming=enable_ifm_streaming,
        enable_wgt_reuse=enable_wgt_reuse,
        pin_ifm_l1=pin_ifm_l1,
        pin_wgt_bias_l1=pin_wgt_bias_l1,
        Com=Com,
        Xom=Xom,
        Cim=Cim,
        wgt_memtile_size=wgt_memtile_size,
        num_ifm_subv=num_ifm_subv,
        prm_memtile_size=prm_memtile_size,
        ifm_memtile_size=ifm_memtile_size,
        ofm_memtile_size=ofm_memtile_size,
        conv_kernel_param_size=conv_kernel_param_size,
        param_subv_size=param_subv_size,
        mt_co_pack=mt_co_pack,
        num_pack_wgt_subv=num_pack_wgt_subv,
        enable_add=enable_add,
        is_xint8=is_xint8,
    )

    return dims


def run_scheduler(
    dims: ConvDims,
    back_end: BackEnd,
    kernel_names: List[str],
    kernel_includes: List[str],
    layer: Layer,
    kernel: Kernel,
) -> None:
    """
    Run the appropriate scheduler (compile_dataflow) function based on layer properties.

    Args:
        dims: ConvDims object with all parameters
        back_end: Backend to use
        kernel_names: List of kernel names
        kernel_includes: List of kernel includes
        layer: Layer object
        kernel: Kernel object
    """
    print(dims)

    enable_add = kernel.enable_add
    enable_L2_IFM = layer.in_act_residency == "L2"
    enable_L2_OFM = layer.out_act_residency == "L2"
    is_X8_split = dims.is_X8_split
    clean_overlay()  # Clean before creating new dataflow
    # Calling conv+add dataflow.
    if dims.enable_add and dims.is_xint8==1:
        conv_y8xc_dataflow_matadd.compile_dataflow(dims, back_end, kernel_names, kernel_includes)
    else:
        conv_L3_dataflow.compile_dataflow(dims, back_end, kernel_names, kernel_includes)

def build_conv(
    dims: ConvDims,
    back_end: BackEnd,
    compile_flags: List[str],
    is_standalone_dwc: bool,
    out_folder: Optional[str] = None,
    dump_trace: bool = False,
) -> None:
    """
    Build the overlay and move files to the output folder.

    Args:
        dims: ConvDims object with all parameters
        back_end: Backend to use
        compile_flags: List of compile flags
        is_standalone_dwc: Whether this is a standalone DWC operation
        out_folder: Output folder for files
        dump_trace: Whether to dump trace
    """
    if is_standalone_dwc:
        host_cpp = "dwc_main.cpp"
    elif (os.name == "nt" and not gcc_available) or (
        gcc_available and waic_config.mode == "release"
    ):
        host_cpp = "main_common.cpp"
    else:
        host_cpp = "conv_main.cpp"

    # Add backend-specific compile flags
    backend_flags = conv_preproc_directives(dims, back_end)
    compile_flags += backend_flags
    build_sim_overlay(back_end, host_cpp, compile_flags, dump_trace)
    # build_sim_overlay(back_end, host_cpp, conv_preproc_directives(dims, back_end))


def conv_build_shape(
    back_end: BackEnd,
    kernel_names: List[str],
    kernel_includes: List[str],
    full_tiling_params: dict,
    layer: Layer,
    kernel: Kernel,
    output_dir: Optional[str] = None,
    is_standalone_dwc: bool = False,
    frontend_only: bool = False,
    dump_trace: bool = False,
):
    """
    Main function to build a convolution operation.

    Args:
        back_end: Backend to use
        kernel_names: List of kernel names
        kernel_includes: List of kernel includes
        full_tiling_params: Complete tiling parameters from JSON
        layer: Layer object
        kernel: Kernel object
        output_dir: Directory to output results
        is_standalone_dwc: Whether this is a standalone DWC operation
        frontend_only: If True, exit after scheduler runs successfully
        dump_trace: Whether to dump trace during build

    Returns:
        None
    """
    # Get the number of configurations
    num_configs = len(full_tiling_params["core_tile_params"].get("subvols", []))
    # Try each configuration until one succeeds
    successful_config = None
    successful_dims = None

    for config_idx in range(num_configs):
        try:
            print(f"Trying tiling configuration {config_idx + 1} of {num_configs}...")

            # Extract specific tiling parameters for this configuration
            tiling_params = extract_tiling_params(full_tiling_params, config_idx)

            # Create overlay with specific mode for this configuration
            overlay = Overlay(
                overlay=tiling_params["overlay_info"]["overlay"],
                op_type=layer.orig_op_type,
                mode=tiling_params["overlay_info"]["mode"],
                overlay_type=tiling_params["overlay_info"]["overlay_type"],
            )

            # Copying the Host cpp and dumping the scheduler outputs to the output directory
            copy_files = ("conv_main.cpp", "dwc_main.cpp")
            if output_dir is not None:
                #NOTE: below is added for local build run(python conv_build.py)
                # if not os.path.exists(output_dir):
                #     os.makedirs(output_dir)
                for file in copy_files:
                    src = os.path.join(CURRDIR, file)
                    if os.path.exists(src):
                        dst = os.path.join(output_dir, file)
                        shutil.copy(src, dst)
                    else:
                        print(f"[WARNING] src file not found - {src}")
                os.chdir(output_dir)

            # Prepare dimensions
            dims = prepare_conv_dims(
                tiling_params, layer, kernel, overlay, is_standalone_dwc
            )

            # Run the scheduler
            run_scheduler(dims, back_end, kernel_names, kernel_includes, layer, kernel)

            print(f"Configuration {config_idx + 1} scheduler succeeded!")

            # Store the successful configuration
            successful_config = config_idx
            successful_dims = dims

            # Break on first successful scheduler
            break
        except AssertionError as ae:
            print(f"Configuration {config_idx + 1} failed: {str(ae)}")
            _, _, tb = sys.exc_info()
            tb_info = traceback.extract_tb(tb)
            filename, line, func, text = tb_info[-1]
            print(
                f"An error occured on line {line} in file {filename} {func} in statement {text}"
            )
            print(f"")
            print(traceback.format_exc())
            print(f"")
            continue

        except Exception as e:
            print(f"Configuration {config_idx + 1} failed: {str(e)}")
            _, _, tb = sys.exc_info()
            tb_info = traceback.extract_tb(tb)
            filename, line, func, text = tb_info[-1]
            print(
                f"An error occured on line {line} in file {filename} {func} in statement {text}"
            )
            print(f"")
            print(traceback.format_exc())
            print(f"")
            continue

    # If we found a successful configuration and frontend_only is False, run build once
    if successful_dims is not None and not frontend_only:
        print(f"Running build with successful configuration {successful_config + 1}...")
        try:
            # Generate compile flags once, since they don't depend on tiling params
            compile_flags = prepare_compile_flags(layer, kernel, back_end)

            # Build the overlay with the successful configuration
            build_conv(
                successful_dims,
                back_end,
                compile_flags,
                is_standalone_dwc,
                output_dir,
                dump_trace,
            )
            print("Build completed successfully")

            return (successful_config, successful_dims)
        except Exception as e:
            print(f"Build failed: {str(e)}")
            # clean_overlay()
            raise RuntimeError(
                f"Failed to build with configuration {successful_config + 1}"
            )
    elif successful_dims is not None and frontend_only:
        print("Skipping build as frontend_only is set")
        return (successful_config, successful_dims)
    else:
        print("All tiling configurations failed!")
        raise RuntimeError("Failed to run scheduler with any tiling configuration")


def extract_fields(file_name):
    with open(file_name, "r") as f:
        data = json.load(f)
    return data


def run_conv_op(
    json_info: str | Dict,
    path=CURRDIR,
    txn_mode: int = 0,
    kernel_d: dict = None,
    frontend_only: bool = False,
    output_dir: Optional[str] = None,
    dump_trace: bool = False,
):
    """
    Run a convolution operation with the given parameters.

    Args:
        json_info: Path to the JSON file or JSON dictionary with tiling parameters
        path: Working directory path
        txn_mode: Transaction mode (0 for ADF, other for TxnHostPatch)
        kernel_d: Kernel definitions
        frontend_only: If True, exit after successful scheduler run but skip the build
        dump_trace: Whether to dump trace

    Returns:
        None
    """
    # Save the current directory before changing it
    original_dir = os.getcwd()

    try:
        os.chdir(path)  # because build system only work on current dir (subprocess)

        full_tiling_params = {}
        if not isinstance(json_info, dict):
            full_tiling_params = extract_fields(json_info)
            output_dir = os.path.dirname(os.path.realpath(json_info))
        else:
            full_tiling_params = json_info

        back_end = BackEnd.Adf if txn_mode == 0 else BackEnd.TxnHostPatch
        layer = Layer(json_dict=full_tiling_params["layer_info"])
        kernel = Kernel(layer=layer)

        data = {}
        if not kernel_d:
            data["kernel_names"] = {}
            kernel_names = ["run_conv_a16w8_qdq"]
            kernel_includes = [
                "super.hh",
                "conv/direct_conv_int16x8_generic/direct_conv_int16x8_generic_wrapper.cc",
            ]
            if "Xint8xXint8xXint8" in layer.op_type:
                kernel_names = ["run_conv_xint8"]
                kernel_includes = ["super.hh", "conv/conv_xint8/run_conv_wrapper.cc"]
                if "add" in layer.op_type:
                    kernel_names += ["run_matadd"]
                    kernel_includes += ["conv/conv_xint8/procyon_matadd/matadd.c"]
            elif "uint8xuint8xuint8" in layer.op_type:
                kernel_names = ["run_conv_a8w8_qdq"]
                kernel_includes = [
                    "super.hh",
                    "conv/direct_conv_int8x8_generic/direct_conv_int8x8_generic_conv_wrapper.cc",
                ]
            for k in kernel_names:
                try:
                    data["kernel_names"][k] = kernel_func_list.index(k)
                except ValueError:
                    print(f"Error: '{k}' not found in the kernel func list!")
        else:
            data["kernel_names"] = {}
            kernel_names = kernel_d["kernel_list"]
            for index, k in enumerate(kernel_names):
                data["kernel_names"][k] = index
            kernel_includes = kernel_d["kernel_include"]
        is_standalone_dwc = layer.is_standalone_dwc
        print(f"DBG: output_dir = {output_dir}")
        dump_trace = (
            True
            if full_tiling_params["additional_flags"]["dump_waves"] == True
            else dump_trace
        )
        ret_val = conv_build_shape(
            back_end,
            data["kernel_names"],
            kernel_includes,
            full_tiling_params,
            layer,
            kernel,
            output_dir,
            is_standalone_dwc,
            frontend_only,
            dump_trace,
        )
        return ret_val
    finally:
        # Restore the original directory after execution completes or if an exception occurs
        os.chdir(original_dir)


def main():
    back_end = BackEnd.Adf
    device = Device("strix")
    Ci, Yi, Xi = (8, 8, 8)  # Input
    Co, Yo, Xo = (16, 8, 8)  # Output
    Ky, Kx = (3, 3)  # Kernel
    Sy, Sx = (1, 1)  # Stride
    Py_b, Px_b, Py_a, Px_a = (
        1,
        1,
        1,
        1,
    )  # (pad_H_start, pad_W_start, pad_H_after, pad_W_after)
    is_standalone_dwc = False

    # Create layer dictionary
    layer_dict = {
        "op_type": "Conv_qdq_uint16xuint8xuint16",
        "in_act_shape": [1, Yi, Xi, Ci],
        "out_act_shape": [1, Yo, Xo, Co],
        "in_datatype": "int16",
        "wgt_datatype": "int8",
        "wgt1_datatype": "int16",
        "out_datatype": "int16",
        "in_bytes": 2,
        "wgt_bytes": 1,
        "wgt1_bytes": 2,
        "out_bytes": 2,
        "attributes": {
            "kernel_shape": [Ky, Kx],
            "strides": [Sy, Sx],
            "pads": [Py_b, Px_b, Py_a, Px_a],
            "group": [1],
        },
        "in_wgt_shape": [Kx, Ky, Ci, Co],
        "in_wgt1_shape": [Co],
        "in_act_residency": "L3",
        "out_act_residency": "L3",
        "is_standalone_dwc": is_standalone_dwc,
    }

    # Create Layer and Kernel objects
    layer = Layer(layer_dict)
    kernel = Kernel(layer)

    # Test the tiling optimization
    tiling_opt = ConvTilingOpt(layer, device, "8x4", kernel, overlay_type="B")
    tiling_params = tiling_opt.find_optimal_tiling()
    output_dir = f"conv_{Ci}x{Yi}x{Xi}_{Co}x{Yo}x{Xo}_{Kx}x{Ky}"

    frontend_only = False  # True to run tiler and scheduler only , False to run the test in adf or txn mode
    txn_mode = 1  # default value is 1 which enables BackEnd.TxnHostPatch
    dump_trace = False  # default value is False to disable vcd trace
    if not frontend_only:
        txn_mode = 0  # 0 is BackEnd.Adf and 1 is BackEnd.TxnHostPatch
        dump_trace = False  # True to enable vcd trace , False to disable vcd trace

    ret_val = run_conv_op(
        tiling_params,
        txn_mode=txn_mode,
        frontend_only=frontend_only,
        dump_trace=dump_trace,
        output_dir=output_dir,
    )


if __name__ == "__main__":
    main()
