import json
import shlex
import subprocess
import sys
import os
import glob
from typing import List

def gen_xclbin(overlay: str, output: str, use_bsub: bool) -> int:
    """
    Generate xclbin binary.
    overlay -- overlay as string, e.g., '4x4' or '8x4'
    output -- absolute path to output directory
    use_bsub -- True: launch build process via bsub on LSF cluster,
                False: run build process as subprocess on the same machine
    """

    def run_build(script_name: str, script_args: List[str], cwd: str) -> int:
        command = [sys.executable, script_name] + script_args
        if use_bsub:
            command = [
                "bsub", "-Is", "-R", "rusage[mem=32768]", "-R", "select[type=X86_64]",
                "-R", "select[osdistro=rhel && (osver=ws8)]", "-cwd", cwd
            ] + command
        print("Cwd:", cwd)
        print("Command:", " ".join(shlex.quote(arg) for arg in command))
        try:
            result = subprocess.run(command, check=True, text=True, cwd=cwd)
            print("Output:", result.stdout)
            return result.returncode
        except subprocess.CalledProcessError as e:
            print("Error occurred:", e.stderr)
            raise RuntimeError('Failed to build xclbin!')

    def process_model_files(pattern, output_subdir):
        script_dir = os.path.dirname(os.path.abspath(__file__))
        target_dir = os.path.abspath(os.path.join(script_dir, output_subdir))
        
        file_list = glob.glob(os.path.join(target_dir, pattern))
        
        if not os.path.exists(target_dir):
            print(f"Directory '{target_dir}' does not exist.")
            return None
        
        if not file_list:
            print(f"No files matching pattern '{pattern}' found in '{target_dir}'.")
            return None

        return file_list[0]  # Return full path to the first matching file

    CURRDIR = os.path.dirname(os.path.abspath(__file__))
    
    pattern = "*_IR_kernel_list.json"
    file_path  = process_model_files(pattern,output)
    script_name = "xclbin_build.py"
    script_path = os.path.join(CURRDIR, script_name)

    if not os.path.exists(script_path):
        print(f"Error: The script '{script_name}' does not exist in the current directory '{CURRDIR}'.")
        sys.exit(1)
    script_args = ["-f", file_path, "-d", output, "-o", overlay]
    exit_code = run_build(script_name, script_args, CURRDIR)
    return exit_code

