"""
Automates updating the 'aie-rt' dependency inside a larger repository.

This script performs the following:
1. Creates a temporary folder <REPO_ROOT>/temp.
2. Clones the 'main_aig' branch of https://gitenterprise.xilinx.com/ai-engine/aie-rt
   into <REPO_ROOT>/temp/aie-rt.
3. Replaces REPO_ROOT/aie-rt/src with the freshly cloned driver/src.
4. Builds AIE-RT depending on the provided argument:
      - linux:   make -f Makefile.Linux → copies generated .so files to build_linux
      - windows: cmake configure + build → copies Debug/ and Include/ to build_windows
5. Updates REPO_ROOT/aie-rt/commit.log with the cloned repo's latest commit hash.
6. Deletes <REPO_ROOT>/temp.
7. Stages and commits the updated REPO_ROOT/aie-rt folder.

Usage:
    python update_aie_rt.py linux
    python update_aie_rt.py windows
"""

import argparse
import os
import shutil
import subprocess
import sys
import stat

AIE_RT_URL = "https://gitenterprise.xilinx.com/ai-engine/aie-rt"
AIE_RT_BRANCH = "main_aig"


def run(cmd, cwd=None):
    """
    Execute a command and raise an exception if it fails.
    """
    print(f"[RUN] {' '.join(cmd)} (cwd={cwd or os.getcwd()})")
    subprocess.run(cmd, cwd=cwd, check=True)


def get_repo_root():
    """
    Returns the directory where this script resides.
    """
    return os.path.dirname(os.path.abspath(__file__))


def clone_aie_rt(temp_dir):
    """
    Clone the AIE-RT repository (main_aig branch) into temp_dir/aie-rt.
    """
    clone_dir = os.path.join(temp_dir, "aie-rt")
    if os.path.exists(clone_dir):
        _rmtree_force(clone_dir)

    run(["git", "clone", "--branch", AIE_RT_BRANCH, "--single-branch",
         AIE_RT_URL, clone_dir])

    return clone_dir


def get_latest_commit_hash(repo_dir):
    """
    Get short hash of HEAD commit inside a cloned repo.
    """
    out = subprocess.run(
        ["git", "rev-parse", "--short", "HEAD"],
        cwd=repo_dir,
        capture_output=True,
        text=True,
        check=True
    )
    return out.stdout.strip()


def replace_src(repo_root, cloned_dir):
    """
    Replace REPO_ROOT/aie-rt/src with the new src from the cloned repo.
    """
    src_src = os.path.join(cloned_dir, "driver", "src")
    src_dst = os.path.join(repo_root, "aie-rt", "src")

    if os.path.exists(src_dst):
        _rmtree_force(src_dst)

    shutil.copytree(src_src, src_dst)


def _rmtree_force(path: str) -> None:
    """Robustly remove a directory tree."""
    def _onerror(func, p, exc_info):
        # func is os.remove / os.rmdir / os.unlink
        # p is the path that failed
        exc = exc_info[1]
        if isinstance(exc, PermissionError):
            try:
                # Make it writeable and try again
                os.chmod(p, stat.S_IWRITE)
                func(p)
            except PermissionError:
                print(f"\033[93m[WARN] Could not delete {p} (still in use).\033[0m")
        else:
            # Re-raise anything that isn't a PermissionError
            raise exc

    if os.path.exists(path):
        shutil.rmtree(path, onerror=_onerror)


def build_linux(cloned_dir, repo_root):
    """
    Build Linux AIE-RT using Makefile.Linux and copy output libs.
    """
    driver_root = os.path.join(cloned_dir, "driver")
    driver_src = os.path.join(driver_root, "src")
    include_src = os.path.join(driver_root, "include")

    dst = os.path.join(repo_root, "aie-rt", "build_linux")
    os.makedirs(dst, exist_ok=True)

    run(["make", "-f", "Makefile.Linux", "controlcode"], cwd=driver_src)

    for lib in ["libxaiengine.so", "libxaiengine.so.3", "libxaiengine.so.3.5"]:
        shutil.copy2(os.path.join(driver_src, lib), os.path.join(dst, lib))

    include_dst = os.path.join(dst, "include")
    if os.path.exists(include_dst):
        _rmtree_force(include_dst)

    if os.path.isdir(include_src):
        shutil.copytree(include_src, include_dst)
    else:
        print("\033[93m[WARN] No include/ dir found in Linux build.\033[0m")


def build_windows(cloned_dir, repo_root):
    """
    Build Windows AIE-RT using CMake, then copy Debug/ and Include/.
    """
    driver_src = os.path.join(cloned_dir, "driver", "src")
    build_dir = os.path.join(driver_src, "build")

    run(["cmake", "-S", ".", "-B", "build"], cwd=driver_src)
    run(["cmake", "--build", "./build", "--parallel"], cwd=driver_src)

    debug_src = os.path.join(build_dir, "Debug")
    include_src = os.path.join(build_dir, "Include")

    if not os.path.isdir(debug_src) or not os.path.isdir(include_src):
        raise RuntimeError("Windows build output missing Debug/ or Include/.")

    dst = os.path.join(repo_root, "aie-rt", "build_windows")
    os.makedirs(dst, exist_ok=True)

    for name, src in [("Debug", debug_src), ("Include", include_src)]:
        dst_dir = os.path.join(dst, name)
        if os.path.exists(dst_dir):
            _rmtree_force(dst_dir)
        shutil.copytree(src, dst_dir)


def update_commit_log(repo_root, commit_hash, target):
    """
    Append the commit hash to REPO_ROOT/aie-rt/commit.md.
    """
    log_file = os.path.join(repo_root, "aie-rt", "commit.md")
    os.makedirs(os.path.dirname(log_file), exist_ok=True)

    line = (
        f"AIE-RT Linux Commit: {commit_hash}\n"
        if target == "linux"
        else f"AIE-RT Windows Commit: {commit_hash}\n"
    )

    with open(log_file, "a", encoding="utf-8") as f:
        f.write(line)


def git_stage_and_commit(repo_root, target):
    """
    Stage and commit the updated aie-rt directory.
    """
    msg = (
        "aie-rt Linux build updated"
        if target == "linux"
        else "aie-rt Windows build updated"
    )

    run(["git", "add", "aie-rt"], cwd=repo_root)
    run(["git", "commit", "-m", msg], cwd=repo_root)


def main():
    """Main script entrypoint."""
    parser = argparse.ArgumentParser()
    parser.add_argument("target", choices=["linux", "windows"])
    target = parser.parse_args().target

    repo_root = get_repo_root()
    temp_dir = os.path.join(repo_root, "temp")

    if os.path.exists(temp_dir):
        _rmtree_force(temp_dir)
    os.makedirs(temp_dir)

    try:
        cloned = clone_aie_rt(temp_dir)
        commit_hash = get_latest_commit_hash(cloned)
        print(f"[INFO] Cloned aie-rt @ {commit_hash}")

        replace_src(repo_root, cloned)

        if target == "linux":
            build_linux(cloned, repo_root)
        else:
            build_windows(cloned, repo_root)

        update_commit_log(repo_root, commit_hash, target)

        _rmtree_force(temp_dir)

        git_stage_and_commit(repo_root, target)

        print("[INFO] Done.")
        print("\033[92m[SUCCESS] Changes have been staged and committed. You only need to push now.\033[0m")

    except Exception as e:
        print(f"\033[91m[ERROR] {e}\033[0m", file=sys.stderr)
        if os.path.exists(temp_dir):
            _rmtree_force(temp_dir)
        sys.exit(1)


if __name__ == "__main__":
    main()
