from typing import List, Dict, Union, Type, Optional
import os
from io import StringIO
from .types import (
    OverlayShape, CoreInstr, AieTile, DataTransfer,
    DmaConnection, CoreConnection,
    BackEnd, CascDir
)

def print_run_layer_compilation_inputs(
    overlay_shape: OverlayShape,
    kernel_names: Union[List[str], Dict[str, int]],
    kernel_includes: List[str],
    core_instrs: Union[List[Type[CoreInstr]], Dict[AieTile, List[Type[CoreInstr]]]],
    memtile_transfers: List[DataTransfer],
    shim_transfers: List[DataTransfer],
    dma_connections: List[DmaConnection],
    back_end: BackEnd = BackEnd.Adf,
    core_stack_addr: int = 57344,
    param_channel_id: int = 1,
    layer_name: str = 'run_dma_layer_config',
    layer_file: str = 'dma.hpp',
    casc_dir: Optional[CascDir] = None,
    core_connections: List[CoreConnection] = [],
):
    buffer = StringIO()
    def out(s=""): buffer.write(s + "\n")

    # ---------- helpers ----------
    def _get_attr(obj, name, alt=None):
        if hasattr(obj, name):
            return getattr(obj, name)
        if isinstance(obj, dict):
            return obj.get(name, alt)
        return alt

    def _dma_channel_str(ch):
        d = getattr(ch, "dir", None)
        cid = getattr(ch, "id", None)
        dname = getattr(d, "name", None) or str(d)
        dname = dname.lower() if isinstance(dname, str) else "unknown"
        return f"{dname}_{cid}"

    def _sync_strategy(ss):
        n = getattr(ss, "name", None) or str(ss)
        return n if n.startswith("SyncStrategy.") else f"SyncStrategy.{n}"

    def _length0(param):
        return param.length_i(0) if hasattr(param, "length_i") else getattr(param, "length", 0)

    def _backend(b):
        n = getattr(b, "name", None) or str(b)
        return n if n.startswith("BackEnd.") else f"BackEnd.{n}"

    def _casc_dir(cd):
        if cd is None:
            return "None"
        n = getattr(cd, "name", None) or str(cd)
        return n if n.startswith("CascDir.") else f"CascDir.{n}"

    def _hex_list(addrs):
        return ", ".join(f"0x{a:x}" for a in addrs)

    def _list_str(xs):
        return ", ".join(str(x) for x in xs)

    def _instr_type(obj):
        return obj.__class__.__name__

    def _all_attrs(obj, exclude_methods=True):
        """Get all attributes of an object, excluding methods if requested"""
        # Class constants to exclude from instance attributes (to avoid duplication)
        class_constants = {
            'BD_CONFIG_OP', 'BUFFER_ACQ_OP', 'BUFFER_REL_OP', 
            'KERNEL_CALL_IN0_IN1_OP', 'KERNEL_CALL_OP', 'LOOP_OP',
            'DISABLE_PONG_ADDR'  # ConfigBuffer specific constant
        }
        
        attrs = []
        for attr_name in dir(obj):
            if attr_name.startswith('_'):
                continue
            if attr_name in class_constants:
                continue  # Skip class constants to avoid duplication
            try:
                attr_value = getattr(obj, attr_name)
                if exclude_methods and callable(attr_value):
                    continue
                attrs.append((attr_name, attr_value))
            except:
                continue
        return attrs

    # ---------- header ----------
    out("\n========== RUN LAYER COMPILATION INPUTS ==========\n")

    # ---------- overlay shape ----------
    cols = _get_attr(overlay_shape, "num_cols")
    rows = _get_attr(overlay_shape, "num_rows")
    out("OVERLAY SHAPE:")
    out(f"  Columns: {cols}")
    out(f"  Rows: {rows}\n")

    # ---------- kernel names ----------
    if isinstance(kernel_names, dict):
        out(f"KERNEL NAMES ({len(kernel_names)}):")
        i = 1
        for k, v in kernel_names.items():
            out(f"  {i}. {k} : {v}")
            i += 1
    else:
        out(f"KERNEL NAMES ({len(kernel_names)}):")
        for i, name in enumerate(kernel_names, 1):
            out(f"  {i}. {name}")
    out()

    # ---------- kernel includes ----------
    out(f"KERNEL INCLUDES ({len(kernel_includes)}):")
    for i, inc in enumerate(kernel_includes, 1):
        out(f"  {i}. {inc}")
    out()

    # ---------- core instructions (per-tile headers like C++) ----------
    if isinstance(core_instrs, dict):
        out(f"CORE INSTRUCTIONS ({sum(len(v) for v in core_instrs.values())}):")
        # Deterministic tile order: sort by str(tile)
        for tile, instr_list in sorted(core_instrs.items(), key=lambda kv: str(kv[0])):
            out(f"  Tile {str(tile)} ({len(instr_list)} instructions):")
            for i, instr in enumerate(instr_list, 1):
                cname = _instr_type(instr)
                out(f"    {i}. Type: {cname}")
                out("       ALL Attributes:")
                out(f"            BD_CONFIG_OP: {getattr(type(instr), 'BD_CONFIG_OP', 3)}")
                out(f"            BUFFER_ACQ_OP: {getattr(type(instr), 'BUFFER_ACQ_OP', 1)}")
                out(f"            BUFFER_REL_OP: {getattr(type(instr), 'BUFFER_REL_OP', 2)}")
                out(f"            KERNEL_CALL_IN0_IN1_OP: {getattr(type(instr), 'KERNEL_CALL_IN0_IN1_OP', 5)}")
                out(f"            KERNEL_CALL_OP: {getattr(type(instr), 'KERNEL_CALL_OP', 4)}")
                out(f"            LOOP_OP: {getattr(type(instr), 'LOOP_OP', 0)}")
                
                # Print all instance attributes for debugging completeness
                all_attrs = _all_attrs(instr)
                if all_attrs:
                    for attr_name, attr_value in all_attrs:
                        if attr_name in ['dma_channel']:
                            out(f"            {attr_name}: {_dma_channel_str(attr_value)}")
                        elif attr_name in ['ping_addr', 'pong_addr'] and attr_value is not None:
                            out(f"            {attr_name}: 0x{attr_value:x}")
                        elif attr_name == 'kernel_params':
                            out(f"            {attr_name}_size: {len(attr_value) if attr_value else 0}")
                            if attr_value:
                                hex_str = " ".join(f"{b:02x}" for b in attr_value[:32])
                                if len(attr_value) > 32:
                                    hex_str += "..."
                                out(f"            {attr_name}_hex: {hex_str}")
                        elif attr_name == 'loop_body':
                            out(f"            {attr_name}_length: {len(attr_value) if attr_value else 0}")
                            if attr_value:
                                out(f"            {attr_name}_details:")
                                for lb_idx, lb_instr in enumerate(attr_value, 1):
                                    lb_name = _instr_type(lb_instr)
                                    out(f"              {lb_idx}. Type: {lb_name}")
                                    # Print key attributes of loop body instructions  
                                    if hasattr(lb_instr, 'dma_channel'):
                                        out(f"                 dma_channel: {_dma_channel_str(lb_instr.dma_channel)}")
                                    if hasattr(lb_instr, 'buffer_size'):
                                        out(f"                 buffer_size: {lb_instr.buffer_size}")
                                    if hasattr(lb_instr, 'kernel_name'):
                                        out(f"                 kernel_name: {lb_instr.kernel_name}")
                        else:
                            out(f"            {attr_name}: {attr_value}")
    else:
        out(f"CORE INSTRUCTIONS ({len(core_instrs)}):")
        for i, instr in enumerate(core_instrs, 1):
            cname = _instr_type(instr)
            out(f"    {i}. Type: {cname}")
            out("       ALL Attributes:")
            out(f"            BD_CONFIG_OP: {getattr(type(instr), 'BD_CONFIG_OP', 3)}")
            out(f"            BUFFER_ACQ_OP: {getattr(type(instr), 'BUFFER_ACQ_OP', 1)}")
            out(f"            BUFFER_REL_OP: {getattr(type(instr), 'BUFFER_REL_OP', 2)}")
            out(f"            KERNEL_CALL_IN0_IN1_OP: {getattr(type(instr), 'KERNEL_CALL_IN0_IN1_OP', 5)}")
            out(f"            KERNEL_CALL_OP: {getattr(type(instr), 'KERNEL_CALL_OP', 4)}")
            out(f"            LOOP_OP: {getattr(type(instr), 'LOOP_OP', 0)}")
            
            # Print all instance attributes for debugging completeness
            all_attrs = _all_attrs(instr)
            if all_attrs:
                for attr_name, attr_value in all_attrs:
                    if attr_name in ['dma_channel']:
                        out(f"            {attr_name}: {_dma_channel_str(attr_value)}")
                    elif attr_name in ['ping_addr', 'pong_addr'] and attr_value is not None:
                        out(f"            {attr_name}: 0x{attr_value:x}")
                    elif attr_name == 'kernel_params':
                        out(f"            {attr_name}_size: {len(attr_value) if attr_value else 0}")
                        if attr_value:
                            hex_str = " ".join(f"{b:02x}" for b in attr_value[:32])
                            if len(attr_value) > 32:
                                hex_str += "..."
                            out(f"            {attr_name}_hex: {hex_str}")
                    elif attr_name == 'loop_body':
                        out(f"            {attr_name}_length: {len(attr_value) if attr_value else 0}")
                        if attr_value:
                            out(f"            {attr_name}_details:")
                            for lb_idx, lb_instr in enumerate(attr_value, 1):
                                lb_name = _instr_type(lb_instr)
                                out(f"              {lb_idx}. Type: {lb_name}")
                                # Print key attributes of loop body instructions  
                                if hasattr(lb_instr, 'dma_channel'):
                                    out(f"                 dma_channel: {_dma_channel_str(lb_instr.dma_channel)}")
                                if hasattr(lb_instr, 'buffer_size'):
                                    out(f"                 buffer_size: {lb_instr.buffer_size}")
                                if hasattr(lb_instr, 'kernel_name'):
                                    out(f"                 kernel_name: {lb_instr.kernel_name}")
                    else:
                        out(f"            {attr_name}: {attr_value}")
    out()

    # ---------- memtile transfers ----------
    out(f"MEMTILE TRANSFERS ({len(memtile_transfers)}):")
    for t_idx, t in enumerate(memtile_transfers, 1):
        out(f"  Transfer {t_idx}:")
        out(f"    Repeat Counts: [{_list_str(t.repeat_counts)}]")
        out(f"    Buffer Tile: {str(t.tile)}")
        out(f"    Buffer Addresses: [{_hex_list(t.buffer_addrs)}]")
        out(f"    Buffer Size: {t.buffer_size}")
        out(f"    Sync Strategy: {_sync_strategy(t.sync_strategy)}")
        out(f"    Reuse Ratio: {getattr(t, 'reuse_ratio', 'N/A')}")
        out(f"    Buffer Split: {getattr(t, 'buffer_split', 'N/A')}")

        out(f"    Write Params ({len(t.write_params)}):")
        for p_idx, p in enumerate(t.write_params, 1):
            out(f"      {p_idx}. Type: {p.__class__.__name__}")
            out(f"         DMA: {str(p.dma)}")
            out(f"         Length: {_length0(p)}")
            
            # Print all TransferParams attributes
            num_reconfig = getattr(p, '_num_reconfig', lambda: 1)()
            out(f"         Num Reconfigurations: {num_reconfig}")
            
            for i in range(num_reconfig):
                if num_reconfig > 1:
                    out(f"         Config {i}:")
                    prefix = "           "
                else:
                    prefix = "         "
                    
                if hasattr(p, 'length_i'):
                    out(f"{prefix}Length: {p.length_i(i)}")
                if hasattr(p, 'offset_i'):
                    out(f"{prefix}Offset: {p.offset_i(i)}")
                if hasattr(p, 'step_i'):
                    step = p.step_i(i)
                    out(f"{prefix}Step: [{_list_str(step)}]")
                if hasattr(p, 'wrap_i'):
                    wrap = p.wrap_i(i)
                    out(f"{prefix}Wrap: [{_list_str(wrap)}]")
                if hasattr(p, 'padding_i'):
                    padding = p.padding_i(i)
                    if padding:
                        padding_str = ", ".join(f"({pad[0]},{pad[1]})" for pad in padding)
                        out(f"{prefix}Padding: [{padding_str}]")
                    else:
                        out(f"{prefix}Padding: []")
                if hasattr(p, 'iter_step_i'):
                    iter_step = p.iter_step_i(i)
                    out(f"{prefix}Iter Step: {iter_step if iter_step is not None else 'None'}")
                if hasattr(p, 'iter_wrap_i'):
                    iter_wrap = p.iter_wrap_i(i)
                    out(f"{prefix}Iter Wrap: {iter_wrap if iter_wrap is not None else 'None'}")
                    
            shim_buf_idx = getattr(p, 'shim_buffer_index', None)
            out(f"         Shim Buffer Index: {shim_buf_idx if shim_buf_idx is not None else 'None'}")

        out(f"    Read Params ({len(t.read_params)}):")
        for p_idx, p in enumerate(t.read_params, 1):
            out(f"      {p_idx}. Type: {p.__class__.__name__}")
            out(f"         DMA: {str(p.dma)}")
            out(f"         Length: {_length0(p)}")
            
            # Print all TransferParams attributes
            num_reconfig = getattr(p, '_num_reconfig', lambda: 1)()
            out(f"         Num Reconfigurations: {num_reconfig}")
            
            for i in range(num_reconfig):
                if num_reconfig > 1:
                    out(f"         Config {i}:")
                    prefix = "           "
                else:
                    prefix = "         "
                    
                if hasattr(p, 'length_i'):
                    out(f"{prefix}Length: {p.length_i(i)}")
                if hasattr(p, 'offset_i'):
                    out(f"{prefix}Offset: {p.offset_i(i)}")
                if hasattr(p, 'step_i'):
                    step = p.step_i(i)
                    out(f"{prefix}Step: [{_list_str(step)}]")
                if hasattr(p, 'wrap_i'):
                    wrap = p.wrap_i(i)
                    out(f"{prefix}Wrap: [{_list_str(wrap)}]")
                if hasattr(p, 'padding_i'):
                    padding = p.padding_i(i)
                    if padding:
                        padding_str = ", ".join(f"({pad[0]},{pad[1]})" for pad in padding)
                        out(f"{prefix}Padding: [{padding_str}]")
                    else:
                        out(f"{prefix}Padding: []")
                if hasattr(p, 'iter_step_i'):
                    iter_step = p.iter_step_i(i)
                    out(f"{prefix}Iter Step: {iter_step if iter_step is not None else 'None'}")
                if hasattr(p, 'iter_wrap_i'):
                    iter_wrap = p.iter_wrap_i(i)
                    out(f"{prefix}Iter Wrap: {iter_wrap if iter_wrap is not None else 'None'}")
                    
            shim_buf_idx = getattr(p, 'shim_buffer_index', None)
            out(f"         Shim Buffer Index: {shim_buf_idx if shim_buf_idx is not None else 'None'}")
    out()

    # ---------- shim transfers ----------
    out(f"SHIM TRANSFERS ({len(shim_transfers)}):")
    for t_idx, t in enumerate(shim_transfers, 1):
        out(f"  Transfer {t_idx}:")
        out(f"    Repeat Counts: [{_list_str(t.repeat_counts)}]")
        out(f"    Buffer Tile: {str(t.tile)}")
        out(f"    Buffer Size: {t.buffer_size}")
        out(f"    Sync Strategy: {_sync_strategy(getattr(t, 'sync_strategy', 'N/A'))}")
        out(f"    Reuse Ratio: {getattr(t, 'reuse_ratio', 'N/A')}")
        out(f"    Buffer Split: {getattr(t, 'buffer_split', 'N/A')}")

        out(f"    Write Params ({len(t.write_params)}):")
        for p_idx, p in enumerate(t.write_params, 1):
            out(f"      {p_idx}. Type: {p.__class__.__name__}")
            out(f"         DMA: {str(p.dma)}")
            out(f"         Length: {_length0(p)}")
            
            # Print all TransferParams attributes
            num_reconfig = getattr(p, '_num_reconfig', lambda: 1)()
            out(f"         Num Reconfigurations: {num_reconfig}")
            
            for i in range(num_reconfig):
                if num_reconfig > 1:
                    out(f"         Config {i}:")
                    prefix = "           "
                else:
                    prefix = "         "
                    
                if hasattr(p, 'length_i'):
                    out(f"{prefix}Length: {p.length_i(i)}")
                if hasattr(p, 'offset_i'):
                    out(f"{prefix}Offset: {p.offset_i(i)}")
                if hasattr(p, 'step_i'):
                    step = p.step_i(i)
                    out(f"{prefix}Step: [{_list_str(step)}]")
                if hasattr(p, 'wrap_i'):
                    wrap = p.wrap_i(i)
                    out(f"{prefix}Wrap: [{_list_str(wrap)}]")
                if hasattr(p, 'padding_i'):
                    padding = p.padding_i(i)
                    if padding:
                        padding_str = ", ".join(f"({pad[0]},{pad[1]})" for pad in padding)
                        out(f"{prefix}Padding: [{padding_str}]")
                    else:
                        out(f"{prefix}Padding: []")
                if hasattr(p, 'iter_step_i'):
                    iter_step = p.iter_step_i(i)
                    out(f"{prefix}Iter Step: {iter_step if iter_step is not None else 'None'}")
                if hasattr(p, 'iter_wrap_i'):
                    iter_wrap = p.iter_wrap_i(i)
                    out(f"{prefix}Iter Wrap: {iter_wrap if iter_wrap is not None else 'None'}")
                    
            shim_buf_idx = getattr(p, 'shim_buffer_index', None)
            out(f"         Shim Buffer Index: {shim_buf_idx if shim_buf_idx is not None else 'None'}")

        out(f"    Read Params ({len(t.read_params)}):")
        for p_idx, p in enumerate(t.read_params, 1):
            out(f"      {p_idx}. Type: {p.__class__.__name__}")
            out(f"         DMA: {str(p.dma)}")
            out(f"         Length: {_length0(p)}")
            
            # Print all TransferParams attributes
            num_reconfig = getattr(p, '_num_reconfig', lambda: 1)()
            out(f"         Num Reconfigurations: {num_reconfig}")
            
            for i in range(num_reconfig):
                if num_reconfig > 1:
                    out(f"         Config {i}:")
                    prefix = "           "
                else:
                    prefix = "         "
                    
                if hasattr(p, 'length_i'):
                    out(f"{prefix}Length: {p.length_i(i)}")
                if hasattr(p, 'offset_i'):
                    out(f"{prefix}Offset: {p.offset_i(i)}")
                if hasattr(p, 'step_i'):
                    step = p.step_i(i)
                    out(f"{prefix}Step: [{_list_str(step)}]")
                if hasattr(p, 'wrap_i'):
                    wrap = p.wrap_i(i)
                    out(f"{prefix}Wrap: [{_list_str(wrap)}]")
                if hasattr(p, 'padding_i'):
                    padding = p.padding_i(i)
                    if padding:
                        padding_str = ", ".join(f"({pad[0]},{pad[1]})" for pad in padding)
                        out(f"{prefix}Padding: [{padding_str}]")
                    else:
                        out(f"{prefix}Padding: []")
                if hasattr(p, 'iter_step_i'):
                    iter_step = p.iter_step_i(i)
                    out(f"{prefix}Iter Step: {iter_step if iter_step is not None else 'None'}")
                if hasattr(p, 'iter_wrap_i'):
                    iter_wrap = p.iter_wrap_i(i)
                    out(f"{prefix}Iter Wrap: {iter_wrap if iter_wrap is not None else 'None'}")
                    
            shim_buf_idx = getattr(p, 'shim_buffer_index', None)
            out(f"         Shim Buffer Index: {shim_buf_idx if shim_buf_idx is not None else 'None'}")
    out()

    # ---------- dma connections ----------
    out(f"DMA CONNECTIONS ({len(dma_connections)}):")
    for c_idx, c in enumerate(dma_connections, 1):
        out(f"  Connection {c_idx}: {str(c)}")
        
        # Read DMA details
        out(f"    Read DMA:")
        out(f"      Tile: {str(c.read_dma.tile)}")
        out(f"      Tile Type: {c.read_dma.tile.type.name}")
        out(f"      Tile Column: {c.read_dma.tile.col}")
        out(f"      Tile Row: {c.read_dma.tile.row}")
        out(f"      Channel: {str(c.read_dma.channel)}")
        out(f"      Channel Direction: {c.read_dma.channel.dir.name}")
        out(f"      Channel ID: {c.read_dma.channel.id}")
        
        # Write DMA details  
        out(f"    Write DMA:")
        out(f"      Tile: {str(c.write_dma.tile)}")
        out(f"      Tile Type: {c.write_dma.tile.type.name}")
        out(f"      Tile Column: {c.write_dma.tile.col}")
        out(f"      Tile Row: {c.write_dma.tile.row}")
        out(f"      Channel: {str(c.write_dma.channel)}")
        out(f"      Channel Direction: {c.write_dma.channel.dir.name}")
        out(f"      Channel ID: {c.write_dma.channel.id}")
        
        # Connection summary
        out(f"    Connection Type: {c.read_dma.tile.type.name} -> {c.write_dma.tile.type.name}")
        out(f"    Data Flow: {c.read_dma.channel.dir.name} -> {c.write_dma.channel.dir.name}")
    out()

    # ---------- core connections ----------
    out(f"CORE CONNECTIONS ({len(core_connections)}):")
    for c in core_connections:
        out(f"  {str(c)}")
    out()

    # ---------- tail ----------
    out(f"LAYER NAME: {layer_name}")
    out(f"LAYER FILE: {layer_file}")
    out(f"CASCADE DIRECTION: {_casc_dir(casc_dir)}")
    out(f"CORE STACK ADDRESS: 0x{core_stack_addr:x}")
    out(f"PARAM CHANNEL ID: {param_channel_id}")
    out(f"BACKEND: {_backend(back_end)}\n")
    out("========== END OF RUN LAYER COMPILATION INPUTS ==========\n")

    # Save to file
    filepath = os.path.join(os.getcwd(), "print_run_layer_compilation.txt")
    with open(filepath, "w") as f:
        f.write(buffer.getvalue())

    # # Print to stdout
    # print(buffer.getvalue(), end="")


def print_task_queue_optimization_inputs(
    shape,
    alloc,
    transfer_buffers,
    transfer_locks,
    transfer_repeats,
    transfer_reuse
):
    """
    Print task queue optimization inputs to file WITHOUT modifying any input data structures.
    Uses safe attribute access to prevent corruption of inputs.
    """
    buffer = StringIO()
    def out(s=""): buffer.write(s + "\n")
    
    def print_separator(title):
        out(f"\n{'='*80}")
        out(f"{title:^80}")
        out(f"{'='*80}")
    
    def safe_getattr(obj, attr, default=None):
        """Safely get attribute without triggering property setters or modifying state"""
        try:
            return getattr(obj, attr, default)
        except:
            return default
    
    def print_buffer_descriptor(bd, indent="  "):
        out(f"{indent}BufferDescriptor:")
        out(f"{indent}  - ID: {safe_getattr(bd, 'id', 'N/A')}")
        out(f"{indent}  - AieDma: {safe_getattr(bd, 'aie_dma', 'N/A')}")
        
        buffer_addr = safe_getattr(bd, 'buffer_addr', None)
        if buffer_addr is not None and isinstance(buffer_addr, int):
            out(f"{indent}  - Buffer Address: 0x{buffer_addr:x}")
        else:
            out(f"{indent}  - Buffer Address: {buffer_addr}")
            
        out(f"{indent}  - Offset: {safe_getattr(bd, '_offset', 'N/A')}")
        out(f"{indent}  - Length: {safe_getattr(bd, '_length', 'N/A')}")
        out(f"{indent}  - Step: {safe_getattr(bd, '_step', 'N/A')}")
        out(f"{indent}  - Wrap: {safe_getattr(bd, '_wrap', 'N/A')}")
        out(f"{indent}  - Padding: {safe_getattr(bd, '_padding', 'N/A')}")
        out(f"{indent}  - Iter Step: {safe_getattr(bd, '_iter_step', 'N/A')}")
        out(f"{indent}  - Iter Wrap: {safe_getattr(bd, '_iter_wrap', 'N/A')}")
        out(f"{indent}  - Lock Enable: {safe_getattr(bd, 'lock_enable', 'N/A')}")
        out(f"{indent}  - Lock Acquire: {safe_getattr(bd, 'lock_acq', 'N/A')}")
        out(f"{indent}  - Lock Acquire Value: {safe_getattr(bd, 'lock_acq_value', 'N/A')}")
        out(f"{indent}  - Lock Release: {safe_getattr(bd, 'lock_rel', 'N/A')}")
        out(f"{indent}  - Lock Release Value: {safe_getattr(bd, 'lock_rel_value', 'N/A')}")
        out(f"{indent}  - Use Next BD: {safe_getattr(bd, 'use_next_bd', 'N/A')}")
        
        next_bd = safe_getattr(bd, 'next_bd', None)
        next_bd_id = safe_getattr(next_bd, 'id', None) if next_bd else None
        out(f"{indent}  - Next BD: {next_bd_id}")
        
        out(f"{indent}  - Packet Enable: {safe_getattr(bd, 'packet_enable', 'N/A')}")
        out(f"{indent}  - Packet ID: {safe_getattr(bd, 'packet_id', 'N/A')}")
        out(f"{indent}  - Is Lock BD: {safe_getattr(bd, 'is_lock_bd', 'N/A')}")
        out(f"{indent}  - Name: {safe_getattr(bd, 'name', 'N/A')}")
        out(f"{indent}  - Fold: {safe_getattr(bd, 'fold', 'N/A')}")
    
    def print_lock(lock, indent="  "):
        out(f"{indent}Lock:")
        out(f"{indent}  - Tile: {safe_getattr(lock, 'tile', 'N/A')}")
        out(f"{indent}  - ID: {safe_getattr(lock, 'id', 'N/A')}")
        out(f"{indent}  - Initial Value: {safe_getattr(lock, 'initial_value', 'N/A')}")
    
    # Print Shape information
    print_separator("OVERLAY SHAPE")
    out(f"Start Column: {safe_getattr(shape, 'start_col', 'N/A')}")
    out(f"Number of Columns: {safe_getattr(shape, 'num_cols', 'N/A')}")
    out(f"Start Row: {safe_getattr(shape, 'start_row', 'N/A')}")
    out(f"Number of Rows: {safe_getattr(shape, 'num_rows', 'N/A')}")
    
    # Print Allocator information
    print_separator("DMA ALLOCATOR STATE")
    out("BD Counters:")
    bd_counter = safe_getattr(alloc, 'bd_counter', {})
    if isinstance(bd_counter, dict):
        for tile_key, counter_list in bd_counter.items():
            if isinstance(counter_list, list):
                used_bds = [i for i, val in enumerate(counter_list) if val > 0]
                out(f"  {tile_key}: {counter_list}")
                if used_bds:
                    out(f"    Used BD IDs: {used_bds}")
            else:
                out(f"  {tile_key}: {counter_list}")
    
    out("\nLock Counters:")
    lock_counters = safe_getattr(alloc, 'lock_counters', {})
    if isinstance(lock_counters, dict):
        for tile, lock_count in lock_counters.items():
            out(f"  {tile}: {lock_count}")
    
    out("\nTask Counters:")
    task_counters = safe_getattr(alloc, 'task_counters', {})
    if isinstance(task_counters, dict):
        for dma, task_count in task_counters.items():
            if task_count > 0:
                out(f"  {dma}: {task_count}")
    
    # Print Transfer Buffers
    print_separator("TRANSFER BUFFERS")
    if isinstance(transfer_buffers, list):
        out(f"Number of Transfer Groups: {len(transfer_buffers)}")
        for i, transfer_group in enumerate(transfer_buffers):
            out(f"\nTransfer Group {i}:")
            if isinstance(transfer_group, list):
                out(f"  Number of Buffer Groups: {len(transfer_group)}")
                for j, buffer_group in enumerate(transfer_group):
                    if isinstance(buffer_group, list):
                        out(f"  Buffer Group {j} (contains {len(buffer_group)} BDs):")
                        for k, bd in enumerate(buffer_group):
                            out(f"    BD {k}:")
                            print_buffer_descriptor(bd, "      ")
                    else:
                        out(f"  Buffer Group {j}: {buffer_group}")
            else:
                out(f"  Transfer Group content: {transfer_group}")
    else:
        out(f"Transfer Buffers: {transfer_buffers}")
    
    # Print Transfer Locks
    print_separator("TRANSFER LOCKS")
    if isinstance(transfer_locks, list):
        out(f"Number of Transfer Lock Groups: {len(transfer_locks)}")
        for i, lock_group in enumerate(transfer_locks):
            if isinstance(lock_group, list):
                out(f"\nTransfer Lock Group {i} (contains {len(lock_group)} locks):")
                for j, lock in enumerate(lock_group):
                    out(f"  Lock {j}:")
                    print_lock(lock, "    ")
            else:
                out(f"\nTransfer Lock Group {i}: {lock_group}")
    else:
        out(f"Transfer Locks: {transfer_locks}")
    
    # Print Transfer Repeats
    print_separator("TRANSFER REPEATS")
    if isinstance(transfer_repeats, list):
        out(f"Number of Transfer Repeat Groups: {len(transfer_repeats)}")
        for i, repeat_group in enumerate(transfer_repeats):
            if isinstance(repeat_group, list):
                out(f"Transfer Repeat Group {i}: {repeat_group}")
                out(f"  Number of phases: {len(repeat_group)}")
                try:
                    out(f"  Total repeats: {sum(repeat_group)}")
                    non_zero_phases = [(idx, val) for idx, val in enumerate(repeat_group) if val > 0]
                    if non_zero_phases:
                        out(f"  Non-zero phases: {non_zero_phases}")
                except:
                    out(f"  Could not calculate totals for repeat group")
            else:
                out(f"Transfer Repeat Group {i}: {repeat_group}")
    else:
        out(f"Transfer Repeats: {transfer_repeats}")
    
    # Print Transfer Reuse
    print_separator("TRANSFER REUSE")
    if isinstance(transfer_reuse, list):
        out(f"Number of Transfer Reuse Values: {len(transfer_reuse)}")
        for i, reuse_ratio in enumerate(transfer_reuse):
            out(f"Transfer {i} Reuse Ratio: {reuse_ratio}")
    else:
        out(f"Transfer Reuse: {transfer_reuse}")
    
    # Print summary statistics (with safe calculations)
    print_separator("SUMMARY STATISTICS")
    try:
        total_bds = 0
        if isinstance(transfer_buffers, list):
            for transfer_group in transfer_buffers:
                if isinstance(transfer_group, list):
                    for buffer_group in transfer_group:
                        if isinstance(buffer_group, list):
                            total_bds += len(buffer_group)
        
        total_locks = 0
        if isinstance(transfer_locks, list):
            for lock_group in transfer_locks:
                if isinstance(lock_group, list):
                    total_locks += len(lock_group)
        
        total_phases = 0
        total_repeats = 0
        if isinstance(transfer_repeats, list):
            for repeat_group in transfer_repeats:
                if isinstance(repeat_group, list):
                    total_phases += len(repeat_group)
                    try:
                        total_repeats += sum(repeat_group)
                    except:
                        pass
        
        out(f"Total Buffer Descriptors: {total_bds}")
        out(f"Total Locks: {total_locks}")
        out(f"Total Phases: {total_phases}")
        out(f"Total Repeats: {total_repeats}")
        
        if isinstance(transfer_reuse, list) and len(transfer_reuse) > 0:
            try:
                avg_reuse = sum(transfer_reuse) / len(transfer_reuse)
                out(f"Average Reuse Ratio: {avg_reuse:.2f}")
            except:
                out(f"Average Reuse Ratio: Could not calculate")
        else:
            out(f"Average Reuse Ratio: N/A")
    except Exception as e:
        out(f"Error calculating summary statistics: {e}")
    
    # Print BD usage by tile type and direction (with safe access)
    print_separator("BD USAGE ANALYSIS")
    try:
        bd_usage = {}
        if isinstance(transfer_buffers, list):
            for transfer_group in transfer_buffers:
                if isinstance(transfer_group, list):
                    for buffer_group in transfer_group:
                        if isinstance(buffer_group, list):
                            for bd in buffer_group:
                                try:
                                    aie_dma = safe_getattr(bd, 'aie_dma', None)
                                    if aie_dma:
                                        tile = safe_getattr(aie_dma, 'tile', None)
                                        channel = safe_getattr(aie_dma, 'channel', None)
                                        if tile and channel:
                                            tile_type = safe_getattr(tile, 'type', None)
                                            direction = safe_getattr(channel, 'dir', None)
                                            if tile_type and direction:
                                                key = f"{tile_type}_{direction}"
                                                bd_usage[key] = bd_usage.get(key, 0) + 1
                                except:
                                    continue
        
        for key, count in bd_usage.items():
            out(f"{key}: {count} BDs")
            
        if not bd_usage:
            out("No BD usage data available")
    except Exception as e:
        out(f"Error analyzing BD usage: {e}")
    
    print_separator("END OF TASK QUEUE OPTIMIZATION INPUTS")
    
    # Save to file
    try:
        filepath = os.path.join(os.getcwd(), "task_queue_optimization_inputs.txt")
        with open(filepath, "w") as f:
            f.write(buffer.getvalue())
        out(f"\nLog saved to: {filepath}")
    except Exception as e:
        out(f"\nError saving log file: {e}")
    
    # Optional: Print to stdout for debugging
    # print(buffer.getvalue(), end="")
