//include helper for older tools versions without aie::pipelined_loop, for example 2024.1 2024.2
#if __AIE_ARCH__ == 21
    #if __AIE_MODEL_VERSION__ >= 11100 
        #define __AIE_API_LOOP_PIPE_HELPER_HPP__
    #endif
#elif __AIE_ARCH__ == 40
    #if __AIE_MODEL_VERSION__ >= 3500 
        #define __AIE_API_LOOP_PIPE_HELPER_HPP__
    #endif
#elif __AIE_ARCH__ == 20
    #if __AIE_MODEL_VERSION__ >= 10600 
        #define __AIE_API_LOOP_PIPE_HELPER_HPP__
    #endif
#elif __AIE_ARCH__ == 22
    #if __AIE_MODEL_VERSION__ >= 10200 
        #define __AIE_API_LOOP_PIPE_HELPER_HPP__
    #endif
#endif

#ifndef __AIE_API_LOOP_PIPE_HELPER_HPP__
#define __AIE_API_LOOP_PIPE_HELPER_HPP__


namespace aie {

/**
  * @brief A structure containing options related to loop iteration peeling
  *
  * To be used with \ref aie::pipelined_loop
  */
struct LoopOptions
{
    /** @brief The number of iterations to peel at the front of the loop*/
    unsigned peel_front = 0;

    /** @brief The number of iterations to peel at the back of the loop*/
    unsigned peel_back  = 0;

    /** @brief Adjust the preamble */
    int preamble_offset = 0;
};

/**
 * @ingroup group_utility_functions
 *
 * @brief Invokes a function object a given number of times.
 *        The pipelining can be controlled by optionally peeling iterations.
 *
 * @tparam MinIters Lower bound on the number of iterations of the loop body
 * @tparam Opts     Options related to peeling loop iterations
 * @param count     Number of iterations
 * @param fn        \parblock
 *                  The callable to pipeline
 *
 *                  ```
 *                  constexpr unsigned MinIters = 8;
 *                  auto loop_body = [&](unsigned idx){ ... };
 *                  aie::pipelined_loop<MinIters, aie::LoopOptions{.peel_front = 2, .peel_back = 1}>(n, loop_body);
 *                  ```
 *                  \endparblock
 */
template<unsigned MinIters, LoopOptions Opts = LoopOptions{}, typename Fn>
__aie_inline
void pipelined_loop(unsigned count, Fn &&fn)
{
    constexpr unsigned total_peel_iters = Opts.peel_front + Opts.peel_back;

    static_assert(total_peel_iters + 1 < MinIters, "Requested peeling exceeds loop range");

    REQUIRES_MSG(count >= total_peel_iters, "Cannot peel more iterations than the loop will be executed");

    if constexpr(MinIters > 1)
    {
        constexpr unsigned peel_front = Opts.peel_front;
        constexpr unsigned peel_back  = Opts.peel_back;

        if constexpr(peel_front > 0)
        {
            unroll_times<peel_front>(fn);
            chess_separator();
        }

#if !AIE_API_NATIVE
        [[using chess: prepare_for_pipelining,
                       min_loop_count(MinIters - total_peel_iters),
                       pipeline_adjust_preamble(Opts.preamble_offset)]]
#endif
        for (unsigned i = peel_front; i < count - peel_back; i++)
        {
            fn(i);
        }

        if constexpr(peel_back > 0)
        {
            chess_separator();
            unroll_times<peel_back>([&](auto i) __aie_inline  {
                fn(count - peel_back + i);
            });
        }
    }
    else
    {
#if !AIE_API_NATIVE
        [[using chess: min_loop_count(MinIters)]]
#endif
        for (unsigned i = 0; i < count; i++) {
            fn(i);
        }
    }
}
}

#endif //AIE_API_LOOP_PIPE_HELPER_HPP