#pragma once

#include <array>
#include <exception>
#include <iostream>
#include <map>
#include <sstream>
#include <stdexcept>
#include <string>
#include <vector>
namespace waic_runner {
    template <typename V>
    static std::ostream& operator<<(std::ostream& os, const std::vector<V>& vec) {
        for (const auto& v : vec) {
            os << v << ", ";
        }
        return os;
    }

    template <typename K, typename V>
    static std::ostream& operator<<(std::ostream& os, const std::map<K, V>& dict) {
        for (const auto& [k, v] : dict) {
            os << k << " : " << v << "\n";
        }
        return os;
    }

    template <typename K, typename V>
    static std::ostream& operator<<(std::ostream& os,
        const std::map<K, std::vector<V>>& dict) {
        for (const auto& [k, v] : dict) {
            os << k << " : " << v << "\n";
        }
        return os;
    }

    static std::string combine_file_line(const std::string& file, size_t line) {
        return "[" + file + ":" + std::to_string(line) + "]";
    }

    template <typename T>
    static T& ptr_get_at(T* arr, size_t sz, size_t idx, const char* name,
        const char* file, size_t line) {

        if (idx >= sz) {
            std::ostringstream oss;
            oss << file << ":" << line << " [ERROR] array out-of-bound access"
                << "\n"
                << "Details - name: " << name << ", size: " << sz << ", idx: " << idx
                << std::endl;
            throw std::runtime_error(oss.str());
        }
        return arr[idx];
    }

    template <typename T>
    static T& vector_get_value_at(std::vector<T>& vec, size_t idx, const char* name,
        const char* file, size_t line) {
        return ptr_get_at(vec.data(), vec.size(), idx, name, file, line);
    }

    template <typename T>
    static const T& vector_get_value_at(const std::vector<T>& vec, size_t idx,
        const char* name, const char* file,
        size_t line) {
        return ptr_get_at(vec.data(), vec.size(), idx, name, file, line);
    }

    template <typename T, size_t N>
    static T& vector_get_value_at(std::array<T, N>& vec, size_t idx,
        const char* name, const char* file, size_t line) {
        return ptr_get_at(vec.data(), N, idx, name, file, line);
    }

    template <typename T, size_t N>
    static const T& vector_get_value_at(const std::array<T, N>& vec, size_t idx,
        const char* name, const char* file,
        size_t line) {
        return ptr_get_at(vec.data(), N, idx, name, file, line);
    }

    template <typename T, size_t N>
    static T& vector_get_value_at(T(&vec)[N], size_t idx, const char* name,
        const char* file, size_t line) {
        return ptr_get_at(&(vec[0]), N, idx, name, file, line);
    }

    template <typename T, size_t N>
    static const T& vector_get_value_at(const T(&vec)[N], size_t idx,
        const char* name, const char* file,
        size_t line) {
        return ptr_get_at(&(vec[0]), N, idx, name, file, line);
    }

    template <typename T>
    static T& c_array_get_value_at(T vec[], size_t N, size_t idx, const char* name,
        const char* file, size_t line) {
        return ptr_get_at(&vec[0], N, idx, name, file, line);
    }

    template <typename T>
    static const T& c_array_get_value_at(const T vec[], size_t N, size_t idx,
        const char* name, const char* file,
        size_t line) {
        return ptr_get_at(&vec[0], N, idx, name, file, line);
    }

    template <typename Container, typename K = typename Container::key_type,
        typename V = typename Container::mapped_type>
    static V& map_get_value_at(Container& container, const K& key, const char* name,
        const char* file, size_t line) {
        auto iter = container.find(key);
        if (iter == container.end()) {
            std::ostringstream oss;
            oss << combine_file_line(file, line) << " [ERROR] Invalid Key Access "
                << "(Container: " << name << ", Key: " << key
                << ", Size: " << container.size() << ")\n";
            throw std::runtime_error(oss.str());
        }
        return iter->second;
    }

    template <typename Container, typename K = typename Container::key_type,
        typename V = typename Container::mapped_type>
    static const V& map_get_value_at(const Container& container, const K& key,
        const char* name, const char* file,
        size_t line) {
        auto iter = container.find(key);
        if (iter == container.end()) {
            std::ostringstream oss;
            oss << combine_file_line(file, line) << " [ERROR] Invalid Key Access "
                << "(Name: " << name << ", Key: " << key
                << ", Size: " << container.size() << ")\n";
            throw std::runtime_error(oss.str());
        }
        return iter->second;
    }

    static std::string cvt_to_string(const std::string& str) { return str; }
    static std::string cvt_to_string(const char* str) { return str; }
    template <typename T> static std::string cvt_to_string(T num) {
        std::ostringstream oss;
        oss << num;
        return oss.str();
    }
    template <typename T> static std::string cvt_to_string(std::vector<T> v) {
        std::ostringstream oss;
        oss << "[ ";
        for (const auto& i : v) {
            oss << cvt_to_string(i) << ", ";
        }
        oss << " ]";
        return oss.str();
    }
    template <typename T, typename U>
    static std::string cvt_to_string(std::map<T, U> m) {
        std::ostringstream oss;
        for (const auto& [k, v] : m) {
            oss << cvt_to_string(k) << " : " << cvt_to_string(v) << "\n";
        }
        return oss.str();
    }
}