#pragma once

#include <chrono>
#include <sstream>
#include <string>

// Timing utilities for performance measurement - matching test_runtime.cpp
#define GET_TIMESTAMP() std::chrono::high_resolution_clock::now()
using time_type_us = std::chrono::microseconds;

// Global latency logging system
class GlobalLatencyLogger {
private:
    std::ostringstream log_;
    static GlobalLatencyLogger* instance_;

public:
    static GlobalLatencyLogger& getInstance() {
        static GlobalLatencyLogger instance;
        return instance;
    }
    
    void log(const std::string& label, const std::chrono::microseconds& duration) {
        log_ << label << ": " << duration.count() << " us\n";
    }
    
    void log(const std::string& label, 
             const std::chrono::high_resolution_clock::time_point& start, 
             const std::chrono::high_resolution_clock::time_point& end) {
        auto duration = std::chrono::duration_cast<time_type_us>(end - start);
        log_ << label << ": " << duration.count() << " us\n";
    }
    
    std::string getLog() const {
        return log_.str();
    }
    
    void clear() {
        log_.str("");
        log_.clear();
    }
    
    std::ostringstream& getStream() {
        return log_;
    }
};

// Convenient global access functions
inline void LOG_GLOBAL_LATENCY(const std::string& label, 
                               const std::chrono::high_resolution_clock::time_point& start,
                               const std::chrono::high_resolution_clock::time_point& end) {
    GlobalLatencyLogger::getInstance().log(label, start, end);
}

inline std::ostringstream& GLOBAL_LATENCY_STREAM() {
    return GlobalLatencyLogger::getInstance().getStream();
}

// Macro for latency logging - much more efficient than string concatenation
#define LOG_LATENCY(log_stream, label, start_time, end_time) \
    do { \
        auto duration = std::chrono::duration_cast<time_type_us>(end_time - start_time).count(); \
        *log_stream << label << ": " << duration << " us\n"; \
    } while(0)

// Global latency logging macro - no need for log_stream parameter
#define LOG_GLOBAL_TIMING(label, start_time, end_time) \
    LOG_GLOBAL_LATENCY(label, start_time, end_time)

// RAII Scoped Timer for automatic timing with global logging
class GlobalScopedTimer {
private:
    std::chrono::high_resolution_clock::time_point start_;
    std::string label_;
    
public:
    GlobalScopedTimer(const std::string& label) 
        : label_(label), start_(std::chrono::high_resolution_clock::now()) {}
    
    ~GlobalScopedTimer() {
        auto end = std::chrono::high_resolution_clock::now();
        auto duration = std::chrono::duration_cast<time_type_us>(end - start_).count();
        GLOBAL_LATENCY_STREAM() << label_ << ": " << duration << " us\n";
    }
};

// RAII Scoped Timer for automatic timing
class ScopedTimer {
private:
    std::chrono::high_resolution_clock::time_point start_;
    std::ostringstream* log_;
    std::string label_;
    
public:
    ScopedTimer(std::ostringstream* log, const std::string& label) 
        : log_(log), label_(label), start_(std::chrono::high_resolution_clock::now()) {}
    
    ~ScopedTimer() {
        if (log_) {
            auto end = std::chrono::high_resolution_clock::now();
            auto duration = std::chrono::duration_cast<time_type_us>(end - start_).count();
            *log_ << label_ << ": " << duration << " us\n";
        }
    }
};

// Convenient macros for scoped timing
#define GLOBAL_SCOPED_TIMING(label) GlobalScopedTimer _global_scoped_timer(label)
#define SCOPED_TIMING(log, label) ScopedTimer _scoped_timer(log, label)
