// Copyright (C) 2022 - 2025 Advanced Micro Devices, Inc. All rights reserved.
////////////////////////////////////////////////////////////////////////
#include "host_runtime.h"
#include "json_reader.hpp"
#include "utils.hpp"

#define LOG_VERBOSE()          if (verbose) std::cout

namespace waic_runner {
    json read_json_file(const std::string& json_fname) {
#ifdef _WIN32
        std::filesystem::path file_path(json_fname);
        std::string long_path = make_long_path(file_path);
        std::ifstream ifs(long_path, std::ios::binary);
#else
        std::ifstream ifs(json_fname);
#endif
        if (!ifs.is_open()) {
            throw std::runtime_error("Can not open ctrl meta json file");
        }

        json data;
        try {
            data = json::parse(ifs);
            ifs.close();
        }
        catch (std::exception& e) {
            std::cout << e.what() << std::endl;
            ifs.close();
            throw std::runtime_error("Can not parse json file");
        }
        return data;
    }

    json parse_json(const std::string& json_data) {
        json data;
        try {
            data = json::parse(json_data);
        }
        catch (std::exception& e) {
            std::cout << e.what() << std::endl;
            throw std::runtime_error("Can not parse json file");
        }
        return data;
    }

	std::vector<CtrlPktPatchInfo>
		ext_buf_json_to_ctrlpkt_patch_info(json& data, const std::vector<uint8_t>& ctrl_bin) {

		std::vector<CtrlPktPatchInfo> patch_info;
		try {
			for (const auto& [key, value] : data.at("external_buffers").items()) {
				size_t xrt_id = value.at("xrt_id").template get<std::uint8_t>();
				size_t bo_offset = 0;
				if (value.contains("coalesed_buffers")) {
					for (const auto& buf : value.at("coalesed_buffers")) {
						if (!buf.contains("control_packet_patch_locations")) {
							continue;
						}
						bo_offset = buf.at("offset_in_bytes");
						for (const auto& pi : buf.at("control_packet_patch_locations")) {
							uint64_t val = 0;
							if (pi.at("operation") == "read_add_write") {
								val =
									*(uint64_t*)(ctrl_bin.data() +
										pi.at("offset").template get<std::uint32_t>());
							}
							patch_info.push_back(
								{ pi.at("offset"), pi.at("size"), xrt_id, bo_offset + val });
						}
					}
				}
				else {
					if (!value.contains("control_packet_patch_locations")) {
						continue;
					}
					for (const auto& pi : value.at("control_packet_patch_locations")) {
						uint64_t val =
							*(uint64_t*)(ctrl_bin.data() +
								pi.at("offset").template get<std::uint32_t>());
						patch_info.push_back(
							{ pi.at("offset"), pi.at("size"), xrt_id, bo_offset + val });
					}
				}
			}
		}
		catch (std::exception& e) {
			std::cout << "Failed to parse ctrl pkt meta JSON String" << std::endl;
			std::cout << e.what() << std::endl;
		}

		return patch_info;
	}

    std::vector<CtrlPktPatchInfo>
        extract_ctrlpkt_patch_info(json& data) {

        std::vector<CtrlPktPatchInfo> patch_info;

        for (const auto& ctrlpatch : data.at("ctrl_pkt_patch_info")) {
            patch_info.push_back({ ctrlpatch.at("offset"), ctrlpatch.at("size"),
                                  ctrlpatch.at("xrt_arg_idx"),
                                  ctrlpatch.at("bo_offset") });
        }

        return patch_info;
    }

    static bool isStringInVector(const std::vector<std::string>& vec, const std::string& str) {
        return std::find(vec.begin(), vec.end(), str) != vec.end();
    }

    std::vector<std::string> get_keys(const json& j, const std::vector<std::string>& excludeStr) {
        std::vector<std::string> keys;
        if (j.is_object()) {
            for (auto it = j.begin(); it != j.end(); ++it) {
                if (isStringInVector(excludeStr, it.key()) == 0) {
                    keys.push_back(it.key());
                }
            }
        }
        return keys;
    }
}
