// Copyright (C) 2022 - 2025 Advanced Micro Devices, Inc. All rights reserved.
////////////////////////////////////////////////////////////////////////
#include "utils.hpp"
#include "file_ptr.hpp"
#include <cstdio>
#include <limits>
#include <unordered_map>
#include <iterator>
#include <stdexcept>
#include <filesystem>
#include <cstdint>

#define LOG_VERBOSE_UTIL()          if (verbose) std::cout
namespace waic_runner {
#ifdef _WIN32
    std::string make_long_path(const std::filesystem::path& file_path)
    {
        std::wstring wpath = file_path.wstring();
        if (wpath.rfind(L"\\\\?\\", 0) != 0){
            wpath = L"\\\\?\\" + std::filesystem::absolute(file_path).wstring();
        }
        int size_needed = WideCharToMultiByte(CP_UTF8, 0, wpath.c_str(), -1,
            nullptr, 0, nullptr, nullptr);
        std::string utf8_path(size_needed - 1, 0);
        WideCharToMultiByte(CP_UTF8, 0, wpath.c_str(), -1, utf8_path.data(),
            size_needed, nullptr, nullptr);

        return utf8_path;
    }
#endif

    std::vector<uint8_t> ReadBinaryFile(const std::string& filename, bool verbose)
    {
#ifdef _WIN32
        std::filesystem::path file_path(filename);
        std::string long_path = make_long_path(file_path);
        std::ifstream file(long_path, std::ios::binary);
#else
        std::ifstream file(filename, std::ios::binary);
#endif
        if (!file) {
            throw std::runtime_error("Failed to open file : " +
                filename);
        }
        if (!file.good()) {
            throw std::runtime_error("Failed to load binary: " + filename);
        }
        std::vector<uint8_t> data = std::vector<uint8_t>((std::istreambuf_iterator<char>(file)),
            std::istreambuf_iterator<char>());
        file.close();
        return data;
    }

    void ApplyCtrlPktPatch(uint8_t* pCtrlPktBuf, size_t offset, uint64_t fm_ddr_addr, uint64_t ext_buf_ddr_offset, bool verbose) {
        uint64_t val = *(uint64_t*)(pCtrlPktBuf + offset);
        val &= 0x0000FFFFFFFFFFFF; //48 bit
        LOG_VERBOSE_UTIL() << "bd.address: " << val << "; XRT DDR addr: " << fm_ddr_addr << "; Ext Buff DDR Offset: " << ext_buf_ddr_offset << std::endl;
        val += fm_ddr_addr;
        val += ext_buf_ddr_offset;
        uint32_t lower32BitVal = (uint32_t)val;
        uint16_t higher16BitVal = (uint16_t)((val & 0x0000FFFF00000000) >> 32);
        *(uint32_t*)(pCtrlPktBuf + offset) = lower32BitVal;
        *(uint16_t*)(pCtrlPktBuf + offset + 4) = higher16BitVal;

        // read back and check val
        uint64_t readback_val = *(uint64_t*)(pCtrlPktBuf + offset);
        readback_val &= 0x0000FFFFFFFFFFFF; //48 bit
        if (readback_val != val) {
            std::cout << "values do not match " << val << " " << readback_val << std::endl;
        }
    }

    size_t readDataFromFile(const char* fileName, std::vector<uint32_t>& tensor, bool verbose)
    {
        LOG_VERBOSE_UTIL() << "Loading: " << fileName << std::endl;
        std::ifstream ifs(fileName);
        if (!ifs)
        {
            std::cerr << "Error reading file " << fileName << std::endl;
            exit(EXIT_FAILURE);
        }

        // use number of bytes as hint to reserve memory for vector
        auto sz = ifs.tellg();
        ifs.seekg(0, std::ios::end);
        sz = ifs.tellg() - sz;
        ifs.seekg(0); //rewind

        tensor.reserve(sz >> 2); // assume you on average will have 4 bytes per int

        size_t i = 0;
        std::string hexValue;

        while (ifs >> hexValue) {
            // Convert hexadecimal string to uint32_t value
            uint64_t num = std::stoul(hexValue, nullptr, 16);
            if (num > std::numeric_limits<unsigned>::max()) {
                throw std::out_of_range(std::string(fileName) + " contains number out of range: 0x" + hexValue);
            }
            tensor.push_back((uint32_t)num);
            i++;
        }

        LOG_VERBOSE_UTIL() << "Finished Iterating, Num Elements: " << i << std::endl;

        ifs.close();

        return i * sizeof(uint32_t);
    }

    //void writeDataToFile(const char *fileName, uint32_t *buf, const size_t bufSize)
    //{
    //    std::ofstream ofs(fileName);
    //    if (!ofs)
    //    {
    //        std::cerr << "Error writing file " << fileName << std::endl;
    //        exit(EXIT_FAILURE);
    //    }
    //    for (size_t i = 0; i < bufSize; ++i)
    //    {
    //        ofs << std::hex << std::setw(8) << std::setfill('0') << buf[i] << std::endl;
    //    }
    //
    //    ofs.close();
    //}

    void read_bin_file(std::string filename, char* data) {
        std::ifstream file(filename, std::ios::binary);

        // Check if the file is opened successfully
        if (!file.is_open()) {
            std::cerr << "Error opening file." << std::endl;
            // return 1;
        }

        // Get the file size
        file.seekg(0, std::ios::end);
        std::streampos fileSize = file.tellg();
        file.seekg(0, std::ios::beg);
        file.read(data, fileSize);
        file.close();
    }

    void write_bin_file(std::string filename, char* data, size_t size) {
        std::fstream file;
        file.open(filename, std::ios::out | std::ios::binary);
        file.write(data, size);
    }

    // Align the 'n' to a multiple of 'A'
    // new_n >= n
    // new_n % A = 0
    size_t align_to_next(size_t n, size_t alignment) {
        return ((n + alignment - 1) / alignment) * alignment;
    }

    size_t get_size_of_type(const std::string& type) {
        static const std::unordered_map<std::string, size_t> elem_size{
            {"int8", 1},     {"uint8", 1},   {"int16", 2}, {"uint16", 2},
            {"int32", 4},    {"uint32", 4},  {"int64", 8}, {"uint64", 8},
            {"bfloat16", 2}, {"float32", 4}, {"float", 4}, {"double", 8},
            {"float64", 8} };
        if (elem_size.find(type) == elem_size.end()) {
            throw std::runtime_error("get_size_of_type - Invalid type : " + type);
        }
        auto sz = elem_size.at(type);
        return sz;
    }

    FILE* create_tmpfile() {
#if _WIN32
        FILE* tmp_file = nullptr;
        auto err = tmpfile_s(&tmp_file);
        if (err != 0) {
            throw std::runtime_error("tmpfile_s error");
        }
#else
        FILE* tmp_file = tmpfile();
        if (!tmp_file) {
            throw std::runtime_error("tmpfile error");
        }
#endif
        return tmp_file;
    }

    void dump_to_tmpfile(FILE* file, char* data, size_t size) {
        auto write_size = std::fwrite(data, 1, size, file);
        if (write_size != size) {
            throw std::runtime_error("tmpfile write error");
        }
        fseek64(file, 0, SEEK_SET);
    }

    void save_tmpfile_on_disk(const std::filesystem::path& path, FILE* file, const char* mode) {
        constexpr size_t buffer_size = 64 * 1024;
        char* buffer =
            new char[buffer_size]; // 64KB, on heap to avoid crash the stack
#if _WIN32
        FILE* dest = nullptr;
        errno_t fopen_err = fopen_s(&dest, path.string().c_str(), mode);
        if (fopen_err || !dest) {
#else
        FILE* dest = fopen(path.string().c_str(), mode);
        if (!dest) {
#endif
            throw std::runtime_error("open file failed");
        }
        size_t read_count;
        while ((read_count = fread(buffer, 1, buffer_size, file)) > 0) {
            fwrite(buffer, 1, read_count, dest);
        }
        fseek64(file, 0, SEEK_SET);
        fclose(dest);
        delete[] buffer;
    }

    void save_tmpfile_in_vec(FILE* file, std::vector<uint8_t> &bo) {
        fseek64(file, 0, SEEK_END);
        long size = ftell(file);
        fseek64(file, 0, SEEK_SET);
        std::vector<uint8_t> data(size);
        fread(data.data(), 1, size, file);
        bo.insert(bo.end(), data.begin(), data.end());
    }

    bool check_elf_flow(const std::string & xclbin_path, bool flag_compile) {
        if (flag_compile) {
            // assume WAIC always support elf_flow
            return (true);
        }
        else {
            auto xclbin = xrt::xclbin(xclbin_path);
            std::vector<std::string> kernel_names;
            bool elf_flow = false;
            for (const auto& kernel : xclbin.get_kernels()) {
                const auto kernel_name = kernel.get_name();
                if (kernel_name.rfind("vadd", 0) != 0
                    && kernel_name.rfind("XDP_KERNEL", 0) != 0) {
                    size_t found = kernel_name.find("_ELF");
                    if (found != std::string::npos) { // didn't find
                        elf_flow = true;
                    }
                }
            }

            return (elf_flow);
            // return (false);
        }
    }

    bool check_elf_flow(const xrt::xclbin& xclbin, bool flag_compile) {
        if (flag_compile) {
            // assume WAIC always support elf_flow
            return (true);
        }
        else {
            std::vector<std::string> kernel_names;
            bool elf_flow = false;
            for (const auto& kernel : xclbin.get_kernels()) {
                const auto kernel_name = kernel.get_name();
                if (kernel_name.rfind("vadd", 0) != 0
                    && kernel_name.rfind("XDP_KERNEL", 0) != 0) {
                    size_t found = kernel_name.find("_ELF");
                    if (found != std::string::npos) { // didn't find
                        elf_flow = true;
                    }
                }
            }

            return (elf_flow);
            // return (false);
        }
    }

    std::string get_env_var(const std::string& var,
        const std::string& default_val) {
#ifdef _WIN32
        char* value = nullptr;
        size_t size = 0;
        errno_t err = _dupenv_s(&value, &size, var.c_str());
        std::string result =
            (!err && (value != nullptr)) ? std::string{ value } : default_val;
        free(value);
#else
        const char* value = std::getenv(var.c_str());
        std::string result = (value != nullptr) ? std::string{ value } : default_val;
#endif
        return result;
    }
}
