#!/usr/bin/env python

import sys
import re

# The structure of the main generated by aiecompiler is as follows:
# 
# Control loop1: infinite loop that can be controlled to break via a volatile variable. This is used in AICG devices
#     Control loop2: loop controlled via graph APIs. This is used in simulation and AECG devices
#
#
# This script supports the following transformations:
# - Make control loop2 infinite, so that it does not obey the number of iterations defined at the RT graph level. This
#   is only recommended for HW builds.
# - Removal of control loop1, as it is not used in non-AECG devices.

waic_crt='''
#include <stdlib.h>

extern "C" {
    typedef void (*thunk)();
    extern thunk _ctors_start;   // first element
    extern thunk _ctors_end;     // past-the-last element

    // the compiler is allowed to consider _start and _end as distinct, but they
    // may be overlayed in reality; access one of them through a chess_copy to
    // prevent optimization of the initial _start/_end non-equality test

    static inline void _waic_init() {
        // constructors are called in reverse order of the list
        for (thunk* t = &_ctors_end; t-- != chess_copy(&_ctors_start); )
            (*t)();
    }

    int main(void);

    int _waic_main_init(int argc, char** argv)
    {
        asm volatile(".label __AIE_ARCH_MODEL_VERSION__21011200 global \\n MOVXM sp, #_sp_start_value_DM_stack"
                     : : :);

        _waic_init();

        main(); // run program and stop simulation (never returns)
        return 0;
    }
}
'''

DONE = re.compile(r'^([ ]+)(done\(\);)', re.MULTILINE)
MAIN = re.compile(r'^(int main\(void\) {)', re.MULTILINE)
SUPER_KERNEL = re.compile(r'^([ ]+)(super_kernel.*;)', re.MULTILINE)
CONTROL_LOOP1 = re.compile(r'^([ ]+)(while\(true\))', re.MULTILINE)

def patch_main(work_dir, core):
    cc_filename = '{}/{}/src/{}.cc'.format(work_dir, core, core)

    with open(cc_filename, 'r') as f:
        source = f.read()

    if '__WAIC_PATCHED__' in source:
        return

    ### repl = SUPER_KERNEL.sub(lambda x: x.group(1) + '// ' + x.group(2) + "\n" + x.group(1) + 'super_kernel();', source)
    ### if repl != source:
    ###     source = repl
    ###     print('WAIC: Patched {} to change super_kernel call'.format(cc_filename))

    repl = MAIN.sub(lambda x: waic_crt + '\n' + x.group(1), source)
    if repl != source:
        source = repl
        print('WAIC: Patched {} to introduce custom CRT'.format(cc_filename))

    repl = CONTROL_LOOP1.sub(lambda x: x.group(1) + '// ' + x.group(2) + "\n" + x.group(1) + 'for (int control1 = 0; control1 < 1; ++control1)', source)
    if repl != source:
        print('WAIC: Patched {} to remove control loop'.format(cc_filename))

        repl = DONE.sub(lambda x: x.group(1) + '// ' + x.group(2) + "\n" + x.group(1) + x.group(2) + '\n' + x.group(1) + '__builtin_unreachable();', repl)

        source = repl

    with open(cc_filename, 'w') as f:
        f.write(source)
        f.write('\n\n// __WAIC_PATCHED__')


if __name__ == '__main__':
    work_dir = './Work'
    cores = []

    for i, arg in enumerate(sys.argv):
        if arg == '-C':
            assert (i+1) < len(sys.argv)
            work_dir = sys.argv[i+1]

        elif 'llgen.Makefile' in arg or 'elfgen.Makefile' in arg:
            cores.append(arg[0:3])

    for core in cores:
        patch_main(work_dir, core)
