//-----------------------------------------------------------------------------
//  
//  Copyright (c) 2009 Xilinx Inc.
//
//  Project  : Programmable Wave Generator
//  Module   : dac_spi.v
//  Parent   : wave_gen
//  Children : None
//
//  Description: 
//     This module takes each sample generated by the sample generator and
//     sends it to the digital to analog converter (DAC) using the SPI
//     protocol. Each SPI cycle is 32 clocks long, consisting of an address,
//     command, and 16 bits of data (MSbit first). The DAC on the development
//     boards use only the most significant 12 out of the 16 bit samples, but
//     all 16 bits are transmitted.
//
//  Parameters:
//
//  Local Parameters:
//
//  Notes       : 
//
//  Multicycle and False Paths
//    None.
//
//

`timescale 1ns/1ps


module dac_spi (
  input             clk_tx,       // Clock input
  input             rst_clk_tx,   // Active HIGH reset - synchronous to clk_tx
  input             en_clk_samp,  // Last clk_tx phase of clk_samp
  input      [15:0] samp,         // Sample output
  input             samp_val,     // New sample is available
  // D to A converter outputs
  output            spi_clk_o,    // Clock for SPI - generated by DDR flop
  output reg        spi_mosi_o,   // SPI master-out-slave-in data bit
  output reg        dac_cs_n_o,   // Chip select for DAC
  output reg        dac_clr_n_o   // Active low clear
);


//***************************************************************************
// Parameter definitions
//***************************************************************************

//***************************************************************************
// Reg and Wire declarations
//***************************************************************************

  wire [31:0] data_to_send; // Data to serialize to the DAC

  reg         active;       // Indicates that this module is sending a seq

  reg         old_active;       // The value of active on the last clock
  reg         old_old_active;   // The value of active on the previous clock

  reg  [4:0]  bit_cnt;      // Counts from 30 to 0

  reg         spi_mosi;   // SPI master-out-slave-in data bit
  reg         dac_cs_n;   // Chip select for DAC
  
//***************************************************************************
// Code
//***************************************************************************

  // Generate dac_clr_n - asserted only when internal reset is asserted
  always @(posedge clk_tx)
  begin
    if (rst_clk_tx)
    begin
      dac_clr_n_o     <= 1'b0;
    end
    else
    begin
      dac_clr_n_o     <= 1'b1;
    end // if rst_clk_tx
  end // always 

  // We start a new sequence when samp_val is asserted, and this is the 
  // "active" phase (i.e. the last one) of clk_samp
  assign start_spi = samp_val && en_clk_samp;

  assign data_to_send   = {
    8'b0000_0000,      // Don't care
    4'b0011,           // Write to an Update (Power Up) ch, N
    4'b0000,           // Channel A
    samp               // The sample to send
  };

  // Generate the SPI protocol
  always @(posedge clk_tx)
  begin
    if (rst_clk_tx)
    begin
      active         <= 1'b0;
      old_active     <= 1'b0;
      old_old_active <= 1'b0;
      bit_cnt        <= 5'b00000;
      dac_cs_n       <= 1'b1;      // Active low
      spi_mosi       <= 1'b1;      // Idle at 1'b1
    end
    else
    begin
      old_active     <= active;
      old_old_active <= old_active;
      if (!active)
      begin
        if (start_spi)
        begin
          active       <= 1'b1;             // Go active
          dac_cs_n     <= 1'b0;             // Set CS active
          spi_mosi     <= data_to_send[31]; // Send the first bit of the seq
          bit_cnt      <= 5'd30;            // We have already sent 31
        end // 
        else
        begin
          dac_cs_n     <= 1'b1;             // INACTIVE
          spi_mosi     <= 1'b1;
        end
      end
      else // we are currently active
      begin
        spi_mosi   <= data_to_send[bit_cnt];
        if (bit_cnt == 5'b00000) // This is the last bit
        begin
          active  <= 1'b0;
        end
        else // Not the last bit
        begin
          bit_cnt <= bit_cnt - 1'b1;
        end
      end
    end // if rst
  end // always
  
  // Pipeline the outputs
  always @(posedge clk_tx)
  begin
    if (rst_clk_tx)
    begin
      spi_mosi_o   <= 1'b0;
      dac_cs_n_o   <= 1'b0;
    end
    else
    begin
      spi_mosi_o   <= spi_mosi;
      dac_cs_n_o   <= dac_cs_n;
    end
  end

  
  // Generate the SPI clock using the DDR flop
  // It is based on old_active due to the pipelining of the other
  // signals, and it is extended one extra clock to allow for the clocking
  // of the last bit (old_old_active).
  out_ddr_flop out_ddr_flop_spi_clk_i0 (
    .clk     (clk_tx),
    .rst     (1'b0),
    .d_rise  (1'b0),
    .d_fall  (old_active || old_old_active),
    .q       (spi_clk_o)
  );


endmodule


//<copyright-disclaimer-start>
//  **************************************************************************************************************
//  * © 2026 Advanced Micro Devices, Inc. All rights reserved.                                                   *
//  * DISCLAIMER                                                                                                 *
//  * The information contained herein is for informational purposes only, and is subject to change              *
//  * without notice. While every precaution has been taken in the preparation of this document, it              *
//  * may contain technical inaccuracies, omissions and typographical errors, and AMD is under no                *
//  * obligation to update or otherwise correct this information.  Advanced Micro Devices, Inc. makes            *
//  * no representations or warranties with respect to the accuracy or completeness of the contents of           *
//  * this document, and assumes no liability of any kind, including the implied warranties of noninfringement,  *
//  * merchantability or fitness for particular purposes, with respect to the operation or use of AMD            *
//  * hardware, software or other products described herein.  No license, including implied or                   *
//  * arising by estoppel, to any intellectual property rights is granted by this document.  Terms and           *
//  * limitations applicable to the purchase or use of AMD’s products are as set forth in a signed agreement     *
//  * between the parties or in AMD's Standard Terms and Conditions of Sale. GD-18                               *
//  *                                                                                                            *
//  **************************************************************************************************************
//<copyright-disclaimer-end>
