###########################################################################
## IP Integrator Basic HW Completer Script
###########################################################################

# Load the standard helper file
source -quiet ../../fpgaSupport_scripts/script1.tcl
source -quiet ../../fpgaSupport_scripts/script2.tcl

# Project constants
set verbose 	1
set tcName 	IP_Integrator
set demoOrLab 	completed
set projName 	basicHW

## *********** Step 1: Copying the project ***********
proc copyProject {} {
    variable platform
    variable language
    variable tcName
    variable demoOrLab 
    variable projName 
    variable TRAINING_PATH

    # Set variables for source and destination paths
    set sourcePath $TRAINING_PATH/$tcName/lab
    set destPath $TRAINING_PATH/$tcName/completed

    # Create the destination directories if they don't exist
    file mkdir $destPath

    # Copy the project files to the destination directory
    file copy -force $sourcePath/$platform $destPath
    puts "sourcePath: $sourcePath"
    puts "destPath: $destPath"
}

## *********** Step 2: Creating a new project ***********
proc createProject {} {
    variable platform
    variable language
    variable tcName
    variable demoOrLab 
    variable projName 
    variable TRAINING_PATH

    # Check if the language and platform are defined
    set isLangNotSelected [string compare -nocase $language "undefined"]
    set isPlatNotSelected [string compare -nocase $platform "undefined"]
    
    if {$isLangNotSelected} {
        puts "Please type: use VHDL | Verilog"
        puts "   then rerun the projectCreate"
    } 
    if {$isPlatNotSelected} {
        puts "Please type: use KCU105 | KC705 | KC7xx"
        puts "   then rerun the projectCreate"
    }

    # Create a new project using the selected platform and language
    create_project basicHW  $TRAINING_PATH/$tcName/completed/$platform -part xczu7ev-ffvc1156-2-e
    set_property board_part xilinx.com:zcu104:part0:1.1 [current_project]
}

## *********** Step 3: Creating a Block Design ***********
proc createBlockDesign {} {
    variable platform
    variable language
    variable tcName
    variable demoOrLab 
    variable projName 
    variable TRAINING_PATH

    # Create a new block design named "design_1"
    create_bd_design "design_1"
    update_compile_order -fileset sources_1

    # Add a Zynq UltraScale+ MPSoC block to the design
    startgroup
    create_bd_cell -type ip -vlnv xilinx.com:ip:zynq_ultra_ps_e:3.5 zynq_ultra_ps_e_0
    endgroup

    # Set properties for the Zynq UltraScale+ MPSoC block
    set_property -dict [list \
        CONFIG.PSU__USE__M_AXI_GP0 {1} \
        CONFIG.PSU__USE__M_AXI_GP2 {0} \
    ] [get_bd_cells zynq_ultra_ps_e_0]
    
    # Apply automation for the Zynq UltraScale+ MPSoC block configuration
    apply_bd_automation -rule xilinx.com:bd_rule:zynq_ultra_ps_e -config {apply_board_preset "1"} [get_bd_cells zynq_ultra_ps_e_0]
    set_property CONFIG.PSU__USE__M_AXI_GP1 {0} [get_bd_cells zynq_ultra_ps_e_0]

    # Add and configure additional IP blocks (AXI Timer and GPIO)
    startgroup
    create_bd_cell -type ip -vlnv xilinx.com:ip:axi_timer:2.0 axi_timer_0
    endgroup
    startgroup
    create_bd_cell -type ip -vlnv xilinx.com:ip:axi_gpio:2.0 axi_gpio_0
    endgroup

    # Configure the GPIO blocks for push buttons and LEDs
    copy_bd_objs /  [get_bd_cells {axi_gpio_0}]
    set_property name gpio_push_buttons [get_bd_cells axi_gpio_0]
    set_property name gpio_leds [get_bd_cells axi_gpio_1]
    set_property CONFIG.GPIO_BOARD_INTERFACE {push_button_4bits} [get_bd_cells gpio_push_buttons]
    set_property CONFIG.GPIO_BOARD_INTERFACE {led_4bits} [get_bd_cells gpio_leds]

    # Apply AXI4 and board interface automation to the design
    startgroup
    apply_bd_automation -rule xilinx.com:bd_rule:axi4 -config { Clk_master {Auto} Clk_slave {Auto} Clk_xbar {Auto} Master {/zynq_ultra_ps_e_0/M_AXI_HPM0_FPD} Slave {/axi_timer_0/S_AXI} ddr_seg {Auto} intc_ip {New AXI Interconnect} master_apm {0}} [get_bd_intf_pins axi_timer_0/S_AXI]
    
    apply_bd_automation -rule xilinx.com:bd_rule:board -config { Board_Interface {led_4bits ( LED ) } Manual_Source {Auto}} [get_bd_intf_pins gpio_leds/GPIO]
    
    apply_bd_automation -rule xilinx.com:bd_rule:axi4 -config { Clk_master {Auto} Clk_slave {Auto} Clk_xbar {Auto} Master {/zynq_ultra_ps_e_0/M_AXI_HPM0_FPD} Slave {/gpio_leds/S_AXI} ddr_seg {Auto} intc_ip {New AXI Interconnect} master_apm {0}} [get_bd_intf_pins gpio_leds/S_AXI]
    
    apply_bd_automation -rule xilinx.com:bd_rule:board -config { Board_Interface {push_button_4bits ( Push buttons ) } Manual_Source {Auto}} [get_bd_intf_pins gpio_push_buttons/GPIO]
    
    apply_bd_automation -rule xilinx.com:bd_rule:axi4 -config { Clk_master {Auto} Clk_slave {Auto} Clk_xbar {Auto} Master {/zynq_ultra_ps_e_0/M_AXI_HPM0_FPD} Slave {/gpio_push_buttons/S_AXI} ddr_seg {Auto} intc_ip {New AXI Interconnect} master_apm {0}} [get_bd_intf_pins gpio_push_buttons/S_AXI]
    
    endgroup

    # Regenerate layout, save, and validate the block design
    regenerate_bd_layout
    save_bd_design
    validate_bd_design
}

## *********** Step 4: Generating the HDL Wrapper and Running Implementation ***********
proc createWrapperandimplRun {} {
    variable platform
    variable language
    variable tcName
    variable demoOrLab 
    variable projName 
    variable TRAINING_PATH

    # Generate the HDL wrapper for the block design
    make_wrapper -files [get_files $TRAINING_PATH/$tcName/completed/$platform/basicHW.srcs/sources_1/bd/design_1/design_1.bd] -top

    # Add the wrapper file to the project and generate design targets
    add_files -norecurse $TRAINING_PATH/$tcName/completed/$platform/basicHW.gen/sources_1/bd/design_1/hdl/design_1_wrapper.v
    generate_target all [get_files  $TRAINING_PATH/$tcName/completed/$platform/basicHW.srcs/sources_1/bd/design_1/design_1.bd]

    # Export IP user files and simulation scripts
    export_ip_user_files -of_objects [get_files $TRAINING_PATH/$tcName/completed/$platform/basicHW.srcs/sources_1/bd/design_1/design_1.bd] -no_script -sync -force -quiet
    export_simulation -of_objects [get_files $TRAINING_PATH/$tcName/completed/$platform/basicHW.srcs/sources_1/bd/design_1/design_1.bd] -directory $TRAINING_PATH/$tcName/completed/$platform/basicHW.ip_user_files/sim_scripts -ip_user_files_dir $TRAINING_PATH/$tcName/completed/$platform/basicHW.ip_user_files -ipstatic_source_dir $TRAINING_PATH/$tcName/completed/$platform/basicHW.ip_user_files/ipstatic -lib_map_path [list {modelsim=$TRAINING_PATH/$tcName/completed/$platform/basicHW.cache/compile_simlib/modelsim} {questa=$TRAINING_PATH/$tcName/completed/$platform/basicHW.cache/compile_simlib/questa} {xcelium=$TRAINING_PATH/$tcName/completed/$platform/basicHW.cache/compile_simlib/xcelium} {vcs=$TRAINING_PATH/$tcName/completed/$platform/basicHW.cache/compile_simlib/vcs} {riviera=$TRAINING_PATH/$tcName/completed/$platform/basicHW.cache/compile_simlib/riviera}] -use_ip_compiled_libs -force -quiet

    # Launch the implementation run and generate the bitstream
    launch_runs impl_1 -to_step write_bitstream -jobs 4
    wait_on_run impl_1
    write_hw_platform -fixed -include_bit -force -file $TRAINING_PATH/$tcName/completed/$platform/basicHW.xsa
}

## ********** Running only the steps that are required with Make **************

proc make {stopAt} {

   puts "Running until the step $stopAt"
   #set steps [list S1_openProject S2_observeReport S3_addPriConstr S4_addGenConstr S5_genReport]
   set limit [string tolower $stopAt]
   switch $limit {
      step1  { copyProject }
      step2  { make step1; createProject }
      step3  { make step2; createBlockDesign }
      step4  { make step3; createWrapperandimplRun }
	all    { make step4 }
      default { 
         puts "Call the make proc, Should be make step*" 
			  }	
	}	
}

#<copyright-disclaimer-start>
#  **************************************************************************************************************
#  * © 2026 Advanced Micro Devices, Inc. All rights reserved.                                                   *
#  * DISCLAIMER                                                                                                 *
#  * The information contained herein is for informational purposes only, and is subject to change              *
#  * without notice. While every precaution has been taken in the preparation of this document, it              *
#  * may contain technical inaccuracies, omissions and typographical errors, and AMD is under no                *
#  * obligation to update or otherwise correct this information.  Advanced Micro Devices, Inc. makes            *
#  * no representations or warranties with respect to the accuracy or completeness of the contents of           *
#  * this document, and assumes no liability of any kind, including the implied warranties of noninfringement,  *
#  * merchantability or fitness for particular purposes, with respect to the operation or use of AMD            *
#  * hardware, software or other products described herein.  No license, including implied or                   *
#  * arising by estoppel, to any intellectual property rights is granted by this document.  Terms and           *
#  * limitations applicable to the purchase or use of AMD’s products are as set forth in a signed agreement     *
#  * between the parties or in AMD's Standard Terms and Conditions of Sale. GD-18                               *
#  *                                                                                                            *
#  **************************************************************************************************************
#<copyright-disclaimer-end>
