
//------------------------------------------------------------------------------
`timescale 1 ns/1 ns

`define END_SIM_NS (10**4)

module tb 
    #(parameter SEED_DATA = 8'd97) 
    ();
	
/**********************************************************************************/
/* Internal Parameters */
/**********************************************************************************/
// - Design
localparam HALF_PERIOD_NS = 5;
localparam GSR_DELAY = 100; //ns

// - DSP 
localparam DATA_WIDTH = 8;

// - Parameters for dist_uniform
localparam START 	 = 0;
localparam END 		 = (2**(DATA_WIDTH))-1;	
	
/**********************************************************************************/
/* I/O Ports */
/**********************************************************************************/
// - Input
reg clk_i,rst_i,ce_i;
wire [DATA_WIDTH-1:0] data_i;
// - Output
wire [DATA_WIDTH-1:0] data_o; 
wire dv_o;
/**********************************************************************************/
/* Internal Signals */
/**********************************************************************************/
// - Inputs
reg [DATA_WIDTH-1:0] data;
reg ce;    
reg flag_ce;                                                                               
// - Random
integer seed_data = SEED_DATA;		                                                                                                          
// - Error
reg error_flag;
// - Golden model			
reg [DATA_WIDTH-1:0] data_in_golden; 																
wire [DATA_WIDTH-1:0] data_out_golden; 
real square_root;
real real_data;
reg [(DATA_WIDTH/2)-1:0] integer_part;
reg [(DATA_WIDTH/2)-1:0] fractional_part;
real tmp;
/**********************************************************************************/
/* DUT */
/**********************************************************************************/
top 
    DUV (
          .clk_i(clk_i),
          .rst_i(rst_i),
          .ce_i(ce_i),
          .data_i(data_i),
          .data_o(data_o),
          .dv_o(dv_o)
        );
            
/**********************************************************************************/
/* Initial Blocks */
/**********************************************************************************/
// - Inputs/Internal Signals
initial begin
  // - DUV inputs
  data  = 0;
  ce = 1'b0;
  flag_ce = 1'b0;
  // - Error
  error_flag = 1'b0;
  // - Golden model
  data_in_golden = 0;
  square_root = 0.0;
  integer_part = 0;
  fractional_part = 0;
  tmp = 0.0;
  // - Clock and Reset  
  clk_i = 1'b0;
  rst_i = 1'b1;
  #(GSR_DELAY) rst_i = 1'b0; 
end

// - End of Simulation
initial begin
    #(`END_SIM_NS) 
    if(error_flag) begin
        $display("%c[1;31m",27);
        $display("SIMULATION_FAILED: Unexpected results");
        $display("%c[0m",27); 
    end else begin
        $display("%c[1;32m",27); 
        $display("SIMULATION_PASS: No unexpected results reported");
        $display("%c[0m",27);
    end
    $finish;
end

/**********************************************************************************/
/* Clock */
/**********************************************************************************/
always 
    #HALF_PERIOD_NS clk_i = ~clk_i;
/**********************************************************************************/
/* Stimuli */
/**********************************************************************************/
always @(posedge clk_i) begin 
	if(rst_i) 
		data <= {{(DATA_WIDTH-3){1'b0}},3'b100};
	else
		data <= $dist_uniform(seed_data,START,END);
end

always @(posedge clk_i) begin 
  if(rst_i) begin
    ce <= 1'b0;
    flag_ce <= 1'b0;
  end else begin
    if(flag_ce==1'b0) begin
      ce <= 1'b1;
      flag_ce <= 1'b1;
    end else begin
      ce <= 1'b0;
    end
  end
end


/**********************************************************************************/
/* Redirect Inputs to DSP */
/**********************************************************************************/
assign data_i = data;
assign ce_i = ce | dv_o;
/**********************************************************************************/
/* Golden Model */
/**********************************************************************************/
always @ (posedge clk_i) begin
  if(rst_i) begin
    data_in_golden <= 0;
  end else if(ce_i) begin
    data_in_golden <= data;
  end
end 

always @ (data_in_golden) begin
  integer_part = 0;
  fractional_part = 0;
  tmp = 0.0;
  real_data = data_in_golden;
  square_root = $sqrt(real_data);
  while(square_root>=1.0) begin
    square_root = square_root - 1.0;
    integer_part = integer_part + 1;
  end
  tmp = square_root * 2.0;
  for(integer i=0;i<(DATA_WIDTH/2);i=i+1) begin
    if(tmp>=1.0) begin
      fractional_part[(DATA_WIDTH/2)-1-i] = 1'b1;
      tmp = tmp - 1.0;
    end else begin
      fractional_part[(DATA_WIDTH/2)-1-i] = 1'b0;
    end
    tmp = tmp * 2.0;
  end
end

assign data_out_golden = {integer_part,fractional_part};
/**********************************************************************************/
/* Self Checking Logic */
/**********************************************************************************/
always @ (posedge clk_i) begin
	if(dv_o) begin
		if(data_out_golden !== data_o)
			error_flag <= 1'b1;
	end
end


endmodule

//<copyright-disclaimer-start>
//  **************************************************************************************************************
//  * © 2026 Advanced Micro Devices, Inc. All rights reserved.                                                   *
//  * DISCLAIMER                                                                                                 *
//  * The information contained herein is for informational purposes only, and is subject to change              *
//  * without notice. While every precaution has been taken in the preparation of this document, it              *
//  * may contain technical inaccuracies, omissions and typographical errors, and AMD is under no                *
//  * obligation to update or otherwise correct this information.  Advanced Micro Devices, Inc. makes            *
//  * no representations or warranties with respect to the accuracy or completeness of the contents of           *
//  * this document, and assumes no liability of any kind, including the implied warranties of noninfringement,  *
//  * merchantability or fitness for particular purposes, with respect to the operation or use of AMD            *
//  * hardware, software or other products described herein.  No license, including implied or                   *
//  * arising by estoppel, to any intellectual property rights is granted by this document.  Terms and           *
//  * limitations applicable to the purchase or use of AMD’s products are as set forth in a signed agreement     *
//  * between the parties or in AMD's Standard Terms and Conditions of Sale. GD-18                               *
//  *                                                                                                            *
//  **************************************************************************************************************
//<copyright-disclaimer-end>
