/**
 *
 * memory tester for the NoCddr labs
 *
 * exercises two different blocks of memory and ensures that the memory regions are separate
 *
 * Notes:
 *    \n are used instead of \n to prevent double line feeds when using the emulator
 *    don't set nSamples too large as it will both dump a lot of info to the console, but you run the risk of overwriting stuff
 *
 * history:
 *    2023/12/03 - rs - replaced "\r" with "\n" so that the xil_printf prints the results on terminal. 
 *    2023/06/06 - wk - initial based on ddr4_lpdd4.c
 *
**/

#include <stdio.h>
#include "platform_versal.h"
#include "xil_printf.h"
#include "xil_io.h"

#define DDR_BASEADDR XPAR_AXI_NOC_DDR_LOW_1_BASEADDR
#define LPDDR_BASEADDR XPAR_AXI_NOC_DDR_CH_1_BASEADDR

#define nSamples 32		/* number of pieces of data to read/write in each region - must be small! */


// ***** main (start of program) *****
int main()
{
	// initialize the platform
	u64 datum, expected_datum;
	int errors = 0;
	u64 effective_address;
    init_platform_versal();

	// tell the user what's running
    xil_printf("Versal Multiple DDRMCs Example Design\n");
	
	// task 1: write a known pattern to memory region 1
    xil_printf("\nDDR Write Test\n");
	xil_printf("\tAddr \t:  Datum\n");
    for(int i=0; i<nSamples; i++) {							// write a small amount of data
	    effective_address = DDR_BASEADDR + (8 * i) + 0x10;	// compute the target address - offset prevents overwriting of vector table (?)
		datum = (0xAAAA << 16) | (i << 8) | i;				// prefix the top 16 bits with A's and duplicate the index in the bottom two bytes
    	Xil_Out64(effective_address, datum);				// and write the datum into memory on 8 byte boundaries
    	xil_printf("\t0x%05p  :  0x%x\n", effective_address, datum);	// show the user what we're writing into memory
    }

	// task 2: read back the pattern and ensure that it is correct
    xil_printf("\nDDR Read-back Test\n");
    for(int i=0; i<nSamples; i++) {
		effective_address = DDR_BASEADDR + (8 * i) + 0x10;	// compute the target address
		expected_datum = (0xAAAA << 16) | (i << 8) | i;		// what data are we expecting to get?
		datum = Xil_In64(effective_address);				// read from memory
		if (datum != expected_datum) {
			xil_printf("\tRead Error!  0x04%p  :  0x%x\n", effective_address, datum);
			errors++;
		}
	}
	if (errors == 0) {
		xil_printf("\tno read errors from DDR\n");
	} else {
		xil_printf("\tthere were %d errors encountered when reading from DDR\n", errors);
		errors = 0;				// reset the error count
	}


	// task 3: write a different pattern to different offsets in memory region 2
    xil_printf("\nLPDDR Write Test\n");
	xil_printf("\tAddr \t\t:\tDatum\n");
    for(int i=0; i<nSamples; i++) {
		effective_address = LPDDR_BASEADDR + (8 * i) + 0x20;			// compute the target address
		datum = (0xBBBB << 16) | i;										// prefix the top 16 bits with B's and place the index only in the lower byte
    	Xil_Out64(effective_address, datum);							// write the datum into this memory offset on 8 byte boundaries offset by 0x1000
    	xil_printf("\t0x%p  :  0x%x\n", effective_address, datum);   	// show the user what we're writing into memory
    }

	// task 4: read back the data from memory region 2
    xil_printf("\nLPDDR Read-Back Test\n");
    for(int i=0; i<nSamples; i++) {
		effective_address = LPDDR_BASEADDR + (8 * i) + 0x20;			// compute the target address
		expected_datum = (0xBBBB << 16) | i;							// what data are we expecting to get?
		datum = Xil_In64(effective_address);							// read from memory
		if (datum != expected_datum) {
			xil_printf("\tRead Error!  0x%p  :  0x%x\n", effective_address, datum);
			errors++;
		}		
	}
	if (errors == 0) {
		xil_printf("\tno read errors from LPDDR\n");
	} else {
		xil_printf("\tthere were %d errors encountered when reading from DDR\n", errors);
		errors = 0;				// reset the error count
	}
	
	// task 5: but wait, there's more! what happens if the two memory locations are the same?
	//         we must read where the memory *could* have been written to and must ensure that the results are
	//         *not* what we wrote. If there are what we wrote, then we're pointing to the wrong places!
    xil_printf("\nReading the DDR memory to see if the LPDDR stuff got written here\n");
    for(int i=0; i<0x10; i++) {											// only need to look at one "row" to confirm overwrite or not
		effective_address = DDR_BASEADDR + (8 * i) + 0x20;				// compute the target address
		expected_datum = (0xBBBB << 16) | i;							// what data are we expecting to get?
		datum = Xil_In64(effective_address);							// read from memory
		if (datum == expected_datum) {
			xil_printf("\tPossible Overlap Error!  0x%p  :  0x%x\n", effective_address, datum);
			errors++;
		}		
	}
	if (errors == 0) {		// if we have a 100% error rate, then all is well
		xil_printf("\tno apparent overlap errors in DDR from LPDDR write\n");
	} else {
		xil_printf("\t%d pieces of data that should have been written into the LPDDR were found in DDR indicating an overlap\n", errors);
	}
	/** full disclosure: since neither the DDR nor LPDDR blocks were *NOT* inititalized and the memory, while it *USUALLY* powers up to 
	    all zeros or all ones *MAY*; however unlikely, might match the data we're matching against. This is an extrememly low probability 
	    event, so we're pretending that it can't happen.	
	**/

    cleanup_platform_versal();
    return 0;
}



//<copyright-disclaimer-start>
//  **************************************************************************************************************
//  * © 2026 Advanced Micro Devices, Inc. All rights reserved.                                                   *
//  * DISCLAIMER                                                                                                 *
//  * The information contained herein is for informational purposes only, and is subject to change              *
//  * without notice. While every precaution has been taken in the preparation of this document, it              *
//  * may contain technical inaccuracies, omissions and typographical errors, and AMD is under no                *
//  * obligation to update or otherwise correct this information.  Advanced Micro Devices, Inc. makes            *
//  * no representations or warranties with respect to the accuracy or completeness of the contents of           *
//  * this document, and assumes no liability of any kind, including the implied warranties of noninfringement,  *
//  * merchantability or fitness for particular purposes, with respect to the operation or use of AMD            *
//  * hardware, software or other products described herein.  No license, including implied or                   *
//  * arising by estoppel, to any intellectual property rights is granted by this document.  Terms and           *
//  * limitations applicable to the purchase or use of AMD’s products are as set forth in a signed agreement     *
//  * between the parties or in AMD's Standard Terms and Conditions of Sale. GD-18                               *
//  *                                                                                                            *
//  **************************************************************************************************************
//<copyright-disclaimer-end> 
