#########################################################################################################

# History:
#    2024.1    - DY - 2024/07/05 - Created standalone script for NoCtuning
#    2024.2    - OB - 2025/27/01 - Updated script for connectionFinish proc also instantiate the pull-up for the TGs using inline IP
#    2025.2    - OB - 2026/07/01 - Updated script for connectionFinish proc and TGsupport to add inline constant block

#########################################################################################################

variable trainingPath $::env(TRAINING_PATH);

puts "Version 2025.2";

# project constants
variable tcName	          NoCtuning;
variable labName          $tcName;
variable blockDesignName  blkdsgn;
variable periphName       ???;
variable verbose          1;
variable demoOrLab        lab;
variable nTGs               7;

# selectable bandwidths for the traffic generators
variable bw_tg_0 300;      # write only  walking 1 data  x1K
variable bw_tg_1 300;      # read  only  incr            x1K
variable bw_tg_2 300;      # write only  walking 1 data  x1K
variable bw_tg_3 300;      # read  only  incr            x1K
variable bw_tg_4 300;      # write only  walking 1 data  x100
variable bw_tg_5 200;      # read  only  incr            x100
variable bw_tg_6 200;      # read  only  incr            x100


proc projectCreate {} {
create_project -force NoCtuning $::env(TRAINING_PATH)/NoCtuning/lab -part xcvc1902-vsva2197-2MP-e-S;
set_property board_part xilinx.com:vck190:part0:2.2 [current_project];
}

proc blockDesignCreate {} {
create_bd_design blkdsgn;
}

proc TGadd {} { 
     variable verbose;
  # delete any IP that this proc creates from any previous run
   set IPtoRemove [get_bd_cells -quiet {perf_axi_tg_*}];
   delete_bd_objs -quiet $IPtoRemove;
   # add traffic generators
   set instanceRoot perf_axi_tg_;
   for {set i 0} {$i<7} { incr i} {
      set thisInstance $instanceRoot;
      append thisInstance $i;
      create_bd_cell -type ip -vlnv xilinx.com:ip:perf_axi_tg:1.0 $thisInstance;
   }
   if {$verbose} { puts "   done with TGadd"; }
}


proc toHex {decimalNumber} {
   return [format %X $decimalNumber];
}


proc strsame {a b} {
   set comparisonValue [string compare -nocase $a $b];
   if {$comparisonValue == 0} { return 1 } else { return 0 }
}

# proc:  strLastIndex
# descr: searches for the last occurrence of string 'b' in string 'a'
# @remark find string index last final
# @param a - string to search in
# @param b - target string
# @return returns the index of b in a. if b is NOT in a, returns -1
#
proc strLastIndex {a b} {
   set pos [string last $b $a]
   return $pos
}

proc TGconfigure {} {
   variable verbose;
   if {$verbose} { puts "NoCtuning_completer.tcl:TGconfigure"; }
   variable nTGs;
   variable configuration;
   variable csvFile;
   variable bw_tg_0;
   variable bw_tg_1;
   variable bw_tg_2;
   variable bw_tg_3;
   variable bw_tg_4;
   variable bw_tg_5;
   variable bw_tg_6;

   set instanceRoot perf_axi_tg_;
   for {set i 0} {$i<$nTGs} { incr i} {
      set thisInstance $instanceRoot;
      append thisInstance $i;
      if {$verbose} { puts "   configuring traffic generator #$i"; }
      
      set thisTGpropertyList [list ];
      lappend thisTGpropertyList CONFIG.USER_PARAM_SRC_ID $i;
      lappend thisTGpropertyList CONFIG.USER_PERF_TG {NON_SYNTHESIZABLE};
      lappend thisTGpropertyList CONFIG.USER_C_AXI_WRITE_BANDWIDTH {300};
      lappend thisTGpropertyList CONFIG.USER_C_AXI_WDATA_PATTERN {WALKING_1_DATA};
      lappend thisTGpropertyList CONFIG.USER_TRAFFIC_SHAPING_EN {FALSE};
      lappend thisTGpropertyList CONFIG.USER_EN_VIO_STATUS_MONITOR {FALSE};
      # lappend thisTGpropertyList CONFIG.USER_DEST_ID_PORTS {NONE};
      lappend thisTGpropertyList CONFIG.USER_C_AXI_WDATA_WIDTH {256};
      lappend thisTGpropertyList CONFIG.USER_C_AXI_RDATA_WIDTH {256};
      lappend thisTGpropertyList CONFIG.USER_C_AXI_WRITE_SIZE {32};
      lappend thisTGpropertyList CONFIG.USER_C_AXI_READ_SIZE {1};
      lappend thisTGpropertyList CONFIG.USER_C_AXI_WDATA_VALUE {0x0000000000000000000000000000000000000000000000000000000000000000};
      lappend thisTGpropertyList CONFIG.USER_PERF_TG {NON_SYNTHESIZABLE};
      lappend thisTGpropertyList CONFIG.USER_EN_VIO_STATUS_MONITOR {FALSE};
      # lappend thisTGpropertyList CONFIG.USER_DEST_ID_PORTS {NONE};
      
      # TGs 0,2 do writes
      # TGs 1,3 do reads
      # TG  4 does cache write
      # TG  5,6 does cache reads

      # the initial configuration uses only one port on the DDRMC and uses different 32M memory sections as buffers 
      set    baseAddr 0x000000000;
      append baseAddr [toHex [expr $i * 1]]; # removed the *2
      append baseAddr 000000;
      set    highAddr 0x000000000;
      append highAddr [toHex [expr ($i * 1)+0]]; # removed the *2
      append highAddr FFFFFF;
      
      # based on the traffic generator, select a block of memory to target
      # todo: adjust so that each traffic generator bandwidth is selected - not typing 0,2 and 1,3 and 5,6 together
      switch $i {
         0 -
         2 {
            puts "   assigning TG $i to range $baseAddr:$highAddr";
            lappend thisTGpropertyList CONFIG.USER_C_AXI_WRITE_BANDWIDTH $bw_tg_0;
            lappend thisTGpropertyList CONFIG.USER_C_AXI_WRITE_BASEADDR $baseAddr;
            lappend thisTGpropertyList CONFIG.USER_C_AXI_WRITE_HIGHADDR $highAddr;
            lappend thisTGpropertyList CONFIG.USER_C_AXI_TEST_SELECT {write_only};
            lappend thisTGpropertyList CONFIG.USER_C_AXI_NO_OF_WR_TRANS {1000};
           }
         1 -
         3 {
            puts "   assigning TG $i to range $baseAddr:$highAddr";
            lappend thisTGpropertyList CONFIG.USER_C_AXI_READ_BANDWIDTH $bw_tg_1;
            lappend thisTGpropertyList CONFIG.USER_C_AXI_READ_BASEADDR  $baseAddr; 
            lappend thisTGpropertyList CONFIG.USER_C_AXI_READ_HIGHADDR $highAddr; 
            lappend thisTGpropertyList CONFIG.USER_C_AXI_TEST_SELECT {read_only}; 
            lappend thisTGpropertyList CONFIG.USER_C_AXI_NO_OF_RD_TRANS {1000}; 
           }
         4 {
            lappend thisTGpropertyList CONFIG.USER_C_AXI_READ_BANDWIDTH $bw_tg_4;
            lappend thisTGpropertyList CONFIG.USER_C_AXI_TEST_SELECT {write_only};
            lappend thisTGpropertyList CONFIG.USER_C_AXI_NO_OF_WR_TRANS {100};
            lappend thisTGpropertyList CONFIG.USER_C_AXI_READ_BASEADDR  0x0000000040000000;       
            lappend thisTGpropertyList CONFIG.USER_C_AXI_READ_HIGHADDR  0x000000004FFFFFFF;
           }
         5 -
         6 {
            lappend thisTGpropertyList CONFIG.USER_C_AXI_READ_BANDWIDTH $bw_tg_5;
            lappend thisTGpropertyList CONFIG.USER_C_AXI_TEST_SELECT {read_only};
            lappend thisTGpropertyList CONFIG.USER_C_AXI_NO_OF_RD_TRANS {100};
            lappend thisTGpropertyList CONFIG.USER_C_AXI_READ_BASEADDR  0x0000000040000000;       
            lappend thisTGpropertyList CONFIG.USER_C_AXI_READ_HIGHADDR  0x000000004FFFFFFF;
          }
         default { puts "number of traffic generators is limited to 7!"; }
      }
      puts "***traffic generator list: $thisTGpropertyList";
      set_property -dict $thisTGpropertyList [get_bd_cells $thisInstance];
   } 
   
   if {$verbose} { puts "   done with TGconfigure"; }
}

proc latestVersion { IPname } {
   set lastPos [strLastIndex $IPname :]; # strip off everything beyond the third colon (as this contains the version info)
   set IPnameNoVer [string range $IPname 0 $lastPos]

   set listOfAllIP [lsort [get_ipdefs]]
   foreach pieceOfIP $listOfAllIP {
      set lastPos [strLastIndex $pieceOfIP :]; # strip off everything beyond the third colon (as this contains the version info)
      set pieceOfIPnoVer [string range $pieceOfIP 0 $lastPos];
	  # puts "\comparing $IPnameNoVer against $pieceOfIPnoVer ==> $matches"	  
      if {[string compare $pieceOfIPnoVer $IPnameNoVer] == 0} {
        return $pieceOfIP
      }
   }
   puts stderr "Could not find an entry in the ipdefs for $IPname";
   return $IPname;  # if we can't find a match, just return the string that was passed in
}


proc NoCsAdd {} {
   variable verbose;
   if {$verbose} { puts "NoCbasic_completer.tcl:NoCsAdd"; }
   variable nTGs;
   
   # add the NoCs
   set instanceRoot axi_noc_;
   append instanceRoot 0;        # only one NoC for this design
   create_bd_cell -type ip -vlnv [latestVersion xilinx.com:ip:axi_noc:1.0] $instanceRoot;
   if {$verbose} { puts "   done with NoCsAdd"; }
}

# NoCs configure is designed to support different configurations. To avoid confusion regarding plural or singular, NoCconfigure just calls NoCsConfigure
proc NoCconfigure {} {
   variable verbose;
   if {$verbose} { puts "NoCbasic_completer.tcl:NoCconfigure"; }
   NoCsConfigure;
}
proc NoCsConfigure {} {
   variable verbose;
   if {$verbose} { puts "NoCbasic_completer.tcl:NoCsConfigure"; }
   variable nTGs 7;
   variable configuration 0;
   
   set instanceRoot axi_noc_0;
   #set stopLimit [expr ($nTGs+3)/4];
   #for {set i 0} {$i<$stopLimit} {incr i} {
      set i 0;
      set thisInstance $instanceRoot;
      #append thisInstance $i;
      puts "   configuring $thisInstance";
      set nSlaves $nTGs;
      #if {$i == [expr $stopLimit - 1]} {              # if this is the last NoC, then there may be fewer than 4 slaves...
      #   set nSlaves [expr ($nTGs % 4)];
      #   if {$nSlaves == 0} { set nSlaves 4; }
      #}
      # QoS settings will change from the default to the optimized one
      # base configuration that is the same regardless of which configuration is occurring
      set thisNoCpropertyList [list ];
      lappend thisNoCpropertyList CONFIG.NUM_SI  $nSlaves;
      lappend thisNoCpropertyList CONFIG.NUM_MI  {0};
      lappend thisNoCpropertyList CONFIG.NUM_MC  {1};
      lappend thisNoCpropertyList CONFIG.NUM_MCP {1};   # $nSlaves;
      lappend thisNoCpropertyList CONFIG.MC_EN_INTR_RESP {TRUE};
      lappend thisNoCpropertyList CONFIG.MC_CHAN_REGION0 {DDR_LOW0};
      lappend thisNoCpropertyList CONFIG.MC_PRE_DEF_ADDR_MAP_SEL {ROW_BANK_COLUMN_BGO};
      lappend thisNoCpropertyList CONFIG.MC_READ_BANDWIDTH  {3200.0};
      lappend thisNoCpropertyList CONFIG.MC_WRITE_BANDWIDTH {3200.0};
      lappend thisNoCpropertyList CONFIG.MC_INPUT_FREQUENCY0 {400.000};
      lappend thisNoCpropertyList CONFIG.MC_INPUTCLK0_PERIOD {2500};
      lappend thisNoCpropertyList CONFIG.MC0_FLIPPED_PINOUT {false};
      set_property -dict $thisNoCpropertyList [get_bd_cells $thisInstance]; 
                         
      # create the internal connections and set bandwidth and TC          
      for {set j 0} {$j<$nSlaves} {incr j} {
         puts "working on internal connection: $j with configuration $configuration";
         set channel MC_$j;
         set thisPin /;
         append thisPin $thisInstance /S0 $j _AXI;
         if {$configuration == 0} {
            puts "   using initial configuration";
            set channel MC_0;       # force everything onto one DDRMC port
            set  cmd "set_property -dict \[list ";
            switch $j {
               0 -
               2 {
                  set channelSpec "\{read_bw \{0\} write_bw \{1750\} read_avg_burst \{4\} write_avg_burst \{4\}\}"; 
                 }
               1 -
               3 {
                  set channelSpec "\{read_bw \{1750\} write_bw \{0\} read_avg_burst \{4\} write_avg_burst \{4\}\}"; 
                 }
               4 {
                  set channelSpec "\{read_bw \{0\} write_bw \{200\} read_avg_burst \{4\} write_avg_burst \{4\}\}";
                 }
               5 -
               6 {
                  set channelSpec "\{read_bw \{200\} write_bw \{0\} read_avg_burst \{4\} write_avg_burst \{4\}\}"
                 }
              default { puts "number of traffic generators is limited to 7!"; }
            }
            append cmd "CONFIG.CONNECTIONS \{$channel $channelSpec \} ";
            switch $j {
               0 -
               2 {
                  append cmd "CONFIG.W_TRAFFIC_CLASS \{BEST_EFFORT\} ";
                 }
               1 -
               3 {
                  append cmd "CONFIG.R_TRAFFIC_CLASS \{BEST_EFFORT\} ";
                 }
               4 {
                  append cmd "CONFIG.W_TRAFFIC_CLASS \{BEST_EFFORT\} ";
                 }
               5 -
               6 {
                  append cmd "CONFIG.R_TRAFFIC_CLASS \{BEST_EFFORT\} ";
                 }
              default { puts "number of traffic generators is limited to 7!"; }
            }            
            append cmd "\] \[get_bd_intf_pins $thisPin\]";
         } elseif {$configuration == 1}  {
            puts "   using second configuration";
            set channel MC_0;             # still only using one DDRMC port
            set    cmd "set_property -dict \[list ";
            set thisNoCpropertyList [list ];
            switch $j {
               0 -
               2 {
                     set channelSpec "\{read_bw \{0\} write_bw \{1750\} read_avg_burst \{4\} write_avg_burst \{4\}\}"; 
                     append cmd {CONFIG.W_TRAFFIC_CLASS {ISOCHRONOUS}};
                 }
               1 -
               3 {
                     set channelSpec "\{ read_bw \{1750\} write_bw \{0\} read_avg_burst \{4\} write_avg_burst \{4\}\}"; 
                     append cmd {CONFIG.R_TRAFFIC_CLASS {ISOCHRONOUS}};
                 }
               4 {
                    set channelSpec "\{ read_bw \{0\} write_bw \{200\} read_avg_burst \{4\} write_avg_burst \{4\}\}";
                    append cmd {CONFIG.W_TRAFFIC_CLASS {BEST_EFFORT}};
                 }
               5 -
               6 {
                  set channelSpec "\{ read_bw \{200\} write_bw \{0\} read_avg_burst \{4\} write_avg_burst \{4\}\}"
                  append cmd {CONFIG.R_TRAFFIC_CLASS {LOW_LATENCY}};
                 }
              default { puts "number of traffic generators is limited to 7!"; }
            }
            append cmd " CONFIG.CONNECTIONS \{$channel $channelSpec \}";
            append cmd "\] \[get_bd_intf_pins $thisPin\]";        
         } elseif {$configuration == 2}  {
            puts "   using DDRMC multi-port configuration";
            # use three DDRMC ports
            set thisNoCpropertyList [list ];
            lappend thisNoCpropertyList CONFIG.NUM_MC  {1};
            lappend thisNoCpropertyList CONFIG.NUM_MCP {3};   
            set_property -dict $thisNoCpropertyList [get_bd_cells $thisInstance]; 
      
            set channel MC_;       
            if {$j < 2}     { append channel 0; } \
            elseif {$j < 4} { append channel 1; } \
            else            { append channel 2; }            
            set cmd "set_property -dict \[list ";
            switch $j {
               0 -
               2 {
                     set channelSpec "\{read_bw \{0\} write_bw \{1750\} read_avg_burst \{4\} write_avg_burst \{4\}\}"; 
                     append cmd {CONFIG.W_TRAFFIC_CLASS {ISOCHRONOUS}};
                 }
               1 -
               3 {
                     set channelSpec "\{ read_bw \{1750\} write_bw \{0\} read_avg_burst \{4\} write_avg_burst \{4\}\}"; 
                     append cmd {CONFIG.R_TRAFFIC_CLASS {ISOCHRONOUS}};
                 }
               4 {
                    set channelSpec "\{ read_bw \{0\} write_bw \{200\} read_avg_burst \{4\} write_avg_burst \{4\}\}";
                    append cmd {CONFIG.W_TRAFFIC_CLASS {BEST_EFFORT}};
                 }
               5 -
               6 {
                  set channelSpec "\{ read_bw \{200\} write_bw \{0\} read_avg_burst \{4\} write_avg_burst \{4\}\}"
                  append cmd {CONFIG.R_TRAFFIC_CLASS {LOW_LATENCY}};
                 }
              default { puts "number of traffic generators is limited to 7!"; }
            }
            append cmd " CONFIG.CONNECTIONS \{$channel $channelSpec \}";
            append cmd "\] \[get_bd_intf_pins $thisPin\]";   
            
            # auto-assign the new addresses  
            assign_bd_address;          
         } elseif {$configuration == 3} {
            # NOT a standalone configuration - it builds upon the last configuration run
            set_property -dict [list CONFIG.MC_READ_BANDWIDTH {6400.0} CONFIG.MC_WRITE_BANDWIDTH {6400.0}] [get_bd_cells axi_noc_0];
         }
         puts "Attempting to execute: $cmd";
         [eval $cmd];
      }
   #}
  
   puts "done with NoCsConfigure";
}

proc NoCsupport {} {
   variable verbose;
   if {$verbose} { puts "NoCbasic_completer.tcl:NoCsupport"; }
   variable nTGs;
   
   puts "   create the noc_clk_gen";
   delete_bd_objs [get_bd_cells -quiet noc_clk*];
   apply_bd_automation -rule xilinx.com:bd_rule:axi_noc -config {num_axi_tg "None" num_aximm_ext "None" pl2noc_apm "0" num_axi_bram "None" num_mc "None" noc_clk "New/Reuse Simulation Clock And Reset Generator" }  [get_bd_cells axi_noc_0];
   set_property -dict [list CONFIG.USER_NUM_OF_SYS_CLK 1] get_bd_cells noc_clk_gen
      
   # connect NoC_clk_gen's clk to NoCs
   delete_bd_objs [get_bd_cells -quiet *clk_wiz*];
   apply_bd_automation -rule xilinx.com:bd_rule:clkrst -config { Clk {New Clocking Wizard} Freq {100} Ref_Clk0 {} Ref_Clk1 {} Ref_Clk2 {}}  [get_bd_pins noc_clk_gen/axi_clk_in_0];
   apply_bd_automation -rule xilinx.com:bd_rule:board -config { Board_Interface {Custom} Manual_Source {Auto}}  [get_bd_intf_pins noc_clk_gen/SYS_CLK0_IN];
   
   # connect clk_wiz's clock input and reset to outside world
   apply_bd_automation -rule xilinx.com:bd_rule:board -config { Clk {New External Port} Manual_Source {Auto}}  [get_bd_pins clk_wiz/clk_in1];
   apply_bd_automation -rule xilinx.com:bd_rule:board -config { Manual_Source {New External Port (ACTIVE_HIGH)}}  [get_bd_pins clk_wiz/reset];

   # connect th rst_clk_wiz's external reset to the NoC clk gen's reset
   apply_bd_automation -rule xilinx.com:bd_rule:board -config { Manual_Source {Auto}}  [get_bd_pins rst_clk_wiz_100M/ext_reset_in];
   
   # fix the NoC clock - this is where the problem appears to be TODO: verify and fix
   set_property -dict [list CONFIG.MC_INPUT_FREQUENCY0 {400.000} CONFIG.MC_INPUTCLK0_PERIOD {2500} CONFIG.MC_F1_LPDDR4_MR1 {0x000} CONFIG.MC_F1_LPDDR4_MR2 {0x000}] [get_bd_cells axi_noc_0]
   
   puts "   done with NoCsupport";
}


proc NoCsupport {} {
   variable verbose;
   if {$verbose} { puts "NoCbasic_completer.tcl:NoCsupport"; }
   variable nTGs;
   
   puts "   create the noc_clk_gen";
   delete_bd_objs [get_bd_cells -quiet noc_clk*];
   apply_bd_automation -rule xilinx.com:bd_rule:axi_noc -config {num_axi_tg "None" num_aximm_ext "None" pl2noc_apm "0" num_axi_bram "None" num_mc "None" noc_clk "New/Reuse Simulation Clock And Reset Generator" }  [get_bd_cells axi_noc_0];
   set_property -dict [list CONFIG.USER_NUM_OF_SYS_CLK {1}] [get_bd_cells noc_clk_gen];
   
   # connect NoC_clk_gen's clk to NoCs
   delete_bd_objs [get_bd_cells -quiet *clk_wiz*];
   apply_bd_automation -rule xilinx.com:bd_rule:clkrst -config { Clk {New Clocking Wizard} Freq {100} Ref_Clk0 {} Ref_Clk1 {} Ref_Clk2 {}}  [get_bd_pins noc_clk_gen/axi_clk_in_0];
   apply_bd_automation -rule xilinx.com:bd_rule:board -config { Board_Interface {Custom} Manual_Source {Auto}}  [get_bd_intf_pins noc_clk_gen/SYS_CLK0_IN];
  
   # connect clk_wiz's clock input and reset to outside world
   apply_bd_automation -rule xilinx.com:bd_rule:board -config { Clk {New External Port} Manual_Source {Auto}}  [get_bd_pins clk_wiz/clk_in1];
   apply_bd_automation -rule xilinx.com:bd_rule:board -config { Manual_Source {New External Port (ACTIVE_HIGH)}}  [get_bd_pins clk_wiz/reset];

   # connect th rst_clk_wiz's external reset to the NoC clk gen's reset
   apply_bd_automation -rule xilinx.com:bd_rule:board -config { Manual_Source {Auto}}  [get_bd_pins rst_clk_wiz_100M/ext_reset_in];
   
   # fix the NoC clock - this is where the problem appears to be TODO: verify and fix
   set_property -dict [list CONFIG.MC_INPUT_FREQUENCY0 {400.000} CONFIG.MC_INPUTCLK0_PERIOD {2500} CONFIG.MC_F1_LPDDR4_MR1 {0x000} CONFIG.MC_F1_LPDDR4_MR2 {0x000}] [get_bd_cells axi_noc_0]
   
   puts "   done with NoCsupport";
}

proc connectionFinish {} {
   variable verbose;
   if {$verbose} { puts "NoCbasic_completer.tcl:connectionFinish"; }
   variable nTGs;
   
   puts "   connect the traffic generators to the NoCs";
   delete_bd_objs [get_bd_intf_nets perf_axi_tg_*_M_AXI];
   for {set i 0} {$i<$nTGs} {incr i} {   
      set src perf_axi_tg_;            # src is the traffic generator
      append src $i /M_AXI;  
      set dst axi_noc_;                # dst is the traffic generator                 
      #set nocNumber [expr $i / 4];
      set nocNumber 0;
      set portNumber $i;   # [expr $i % 4];
      append dst $nocNumber /S0 $portNumber _AXI ;

      connect_bd_intf_net [get_bd_intf_pins $src] [get_bd_intf_pins $dst];
   }

   puts "   connect the traffic generator's start signal to the constant pullup";
   delete_bd_objs [get_bd_nets ilconstant_0_dout];
   for {set i 0} {$i<$nTGs} { incr i} {
      set dstPin perf_axi_tg_;
      append dstPin $i /axi_tg_start;   
      connect_bd_net [get_bd_pins ilconstant_0/dout] [get_bd_pins $dstPin]   
   }   
   
   puts "   connect the traffic generator's reset signal to the noc_clk_gen";
   disconnect_bd_net -quiet /noc_clk_gen_axi_rst_0_n [get_bd_pins perf_axi_tg*/tg_rst_n];
   for {set i 0} {$i<$nTGs} { incr i} {
      set dstPin perf_axi_tg_;
      append dstPin $i /tg_rst_n;   
      connect_bd_net [get_bd_pins noc_clk_gen/axi_rst_0_n] [get_bd_pins $dstPin];  
   }  
   
   puts "   connect the NoCs to the outside world (DDR)";
   delete_bd_objs [get_bd_intf_nets axi_noc_0_CH0_DDR4_0] [get_bd_intf_ports ddr4_rtl*]
   set nNoCs 0; # [expr ($nTGs+3) / 4];
   set i 0;
   #for {set i 0} {$i<$nNoCs} { incr i} {
      set NoCddrPin axi_noc_;
      append NoCddrPin $i /CH0_DDR4_0;
      apply_bd_automation -rule xilinx.com:bd_rule:board -config { Board_Interface {Custom} Manual_Source {Auto}}  [get_bd_intf_pins $NoCddrPin];
  # }
   
   puts "   connect up the clocks to the traffic generators and to the NoC aclks";
   for {set i 0} {$i<$nTGs} { incr i} {
      set TGclkPin perf_axi_tg_;            # src is the traffic generator
      append TGclkPin $i /clk;
      apply_bd_automation -rule xilinx.com:bd_rule:clkrst -config { Clk {/noc_clk_gen/axi_clk_0 (100 MHz)} Freq {100} Ref_Clk0 {} Ref_Clk1 {} Ref_Clk2 {}}  [get_bd_pins $TGclkPin];
   }
   
   puts "   connecting the NoC clk gen to the NoC";
   delete_bd_objs [get_bd_intf_nets ddr4_dimm1_sma_clk_1];
   delete_bd_objs [get_bd_intf_ports ddr4_dimm1];
   delete_bd_objs [get_bd_intf_ports ddr4_dimm1_sma_clk];
   connect_bd_intf_net [get_bd_intf_pins noc_clk_gen/SYS_CLK0] [get_bd_intf_pins axi_noc_0/sys_clk0];
   
   set_property -dict [list CONFIG.MC_INPUT_FREQUENCY0 {400.000} CONFIG.MC_INPUTCLK0_PERIOD {2500} CONFIG.MC_F1_LPDDR4_MR1 {0x000} CONFIG.MC_F1_LPDDR4_MR2 {0x000}] [get_bd_cells axi_noc_0]
  
   
   regenerate_bd_layout;
}

proc TGsupport {} {
   variable verbose;
   if {$verbose} { puts "NoCbasic_completer.tcl:TGsupport"; }
   
   # delete any IP that this proc creates from any previous run
   set IPtoRemove [get_bd_cells -quiet {xlconstant_*}];
   delete_bd_objs -quiet $IPtoRemove;  
   
   # instantiate the pull-up for the TGs using inline IP
   #create_bd_cell -type ip -vlnv [latestVersion xilinx.com:ip:xlconstant:1.1] xlconstant_0;
   create_bd_cell -type inline_hdl -vlnv [latestVersion xilinx.com:inline_hdl:ilconstant:1.0] ilconstant_0;
 
     
   puts "   done with TGsupport";
}



proc markSignalsForSimulation {} {
   variable verbose;
   if {$verbose} { puts "NoCbasic_completer.tcl:markSignalsForSimulation"; }
   variable nTGs;
   
   for {set i 0} {$i<$nTGs} { incr i} {
      set portName perf_axi_tg_;   
      append portName $i _M_AXI;
      set_property SIM_ATTRIBUTE.MARK_SIM true [get_bd_intf_nets $portName];
   }
   
   puts "   done with markSignalsForSimulation";
}


#<copyright-disclaimer-start>
#  **************************************************************************************************************
#  * © 2026 Advanced Micro Devices, Inc. All rights reserved.                                                   *
#  * DISCLAIMER                                                                                                 *
#  * The information contained herein is for informational purposes only, and is subject to change              *
#  * without notice. While every precaution has been taken in the preparation of this document, it              *
#  * may contain technical inaccuracies, omissions and typographical errors, and AMD is under no                *
#  * obligation to update or otherwise correct this information.  Advanced Micro Devices, Inc. makes            *
#  * no representations or warranties with respect to the accuracy or completeness of the contents of           *
#  * this document, and assumes no liability of any kind, including the implied warranties of noninfringement,  *
#  * merchantability or fitness for particular purposes, with respect to the operation or use of AMD            *
#  * hardware, software or other products described herein.  No license, including implied or                   *
#  * arising by estoppel, to any intellectual property rights is granted by this document.  Terms and           *
#  * limitations applicable to the purchase or use of AMD’s products are as set forth in a signed agreement     *
#  * between the parties or in AMD's Standard Terms and Conditions of Sale. GD-18                               *
#  *                                                                                                            *
#  **************************************************************************************************************
#<copyright-disclaimer-end>



