/******************************************************************************
 *  Copyright (c) 2016, Xilinx, Inc.
 *  SPDX-License-Identifier: BSD-3-Clause
 *
 *****************************************************************************/
/******************************************************************************
 *
 *
 * @file ad7991.c
 *
 * Required functions for Pmod AD2 SKU410-217.
 * The PmodAD2 is an analog-to-digital converter powered by the Analog Devices
 * AD7991. Users may communicate with the board through I2C to configure up to
 * 4 conversion channels at 12 bits of resolution.
 * http://store.digilentinc.com/pmodad2-4-channel-12-bit-a-d-converter/
 *
 * <pre>
 * MODIFICATION HISTORY:
 *
 * Ver   Who  Date     Changes
 * ----- --- ------- -----------------------------------------------
 * 1.00a pp  04/13/16 release
 * 1.00b yrq 05/02/16 changed adc_read methods
 *
 * </pre>
 *
 *****************************************************************************/

#include "xparameters.h"
#include "ad7991.h"
#include "i2c.h"


extern i2c device;

void adc_init(void)
{
    u8 cfgValue;
    u8 WriteBuffer[6];
    
/*  
 *  Set default Configuration Register Values
 *    Channel 3 - Selected for conversion
 *    Channel 2 - Selected for conversion
 *    Channel 1 - Selected for conversion
 *    Channel 0 - Selected for conversion
 *    reference Voltage - Supply Voltage
 *    filtering on SDA and SCL - Enabled
 *    bit Trial Delay - Enabled
 *    sample Interval Delay - Enabled
 */
    cfgValue = (1 << CH3)           |
               (1 << CH2)           |
               (1 << CH1)           |
               (1 << CH0)           |
               (0 << REF_SEL)       |
               (0 << FLTR)          |
               (0 << BitTrialDelay) |
               (0 << SampleDelay);
               
    // Write to the Configuration Register
    WriteBuffer[0]=cfgValue;
    i2c_write(device, AD7991IICAddr, WriteBuffer, 1);
}

/*  
 *  Configure the AD7911 device.
 *    chan3 - CHAN3 bit in control register.
 *    chan2 - CHAN2 bit in control register.
 *    chan1 - CHAN1 bit in control register.
 *    chan0 - CHAN0 bit in control register.
 *    ref - REF bit in control register.
 *    filter - FILTER bit in control register.
 *    bit - BIT bit in control register.
 *    sample - SAMPLE bit in control register.
 */
void adc_config(char chan3, char chan2, char chan1, char chan0, 
                    char ref, char filter, char bit, char sample)
{
    u8 cfgValue;
    u8 WriteBuffer[6];

    // Set Configuration Register Values
    cfgValue = (chan3 << CH3)         | // Read Channel 3
               (chan2 << CH2)         | // Read Channel 2
               (chan1 << CH1)         | // Read Channel 1
               (chan0 << CH0)         | // Read Channel 0
               (ref << REF_SEL)       | // Select external reference / Vcc
               (filter << FLTR)       | // filter IIC Bus
               (bit << BitTrialDelay) | // Delay IIC Commands
               (sample << SampleDelay); // Delay IIC Messages
               
    // Write to the Configuration Register
    WriteBuffer[0]=cfgValue;
    i2c_write(device, AD7991IICAddr, WriteBuffer, 1);
}

int adc_read_raw()
{
    int rxData;
    u8 rcvbuffer[2];

    // Read data from AD7991
    i2c_read(device, AD7991IICAddr, rcvbuffer, 2);
    // The first byte is MSB, while the second byte is LSB
    rxData = ((rcvbuffer[0] << 8) | rcvbuffer[1]);

    // Process read voltage value
    return (rxData & BitMask);
    /*
     * We can also check whether received data from AD7991 is correct 
     * by checking the first 2 Leading zeros.
     */
}

float adc_read_voltage(u32 vref)
{
    // Process read voltage value
    return (float)(adc_read_raw() * vref / 4096.0);
}

